<?php

use bff\db\Dynprops;
use bff\img\Thumbnail;

abstract class ItemsBase extends Module implements IModuleWithSvc
{
    /** @var ItemsModel */
    public $model = null;
    protected $securityKey = '9a55ae38a37d260c3b41a1X4e30ea915';

    # Тип пользователей
    const USERTYPE_MEMBER = 1;
    const USERTYPE_AGENT  = 2;

    # Тип отзыва
    const OPINION_STATUS_NEUTRAL  = 1; # нейтральный
    const OPINION_STATUS_NEGATIVE = 2; # отрицательный
    const OPINION_STATUS_POSITIVE = 3; # положительный

    # ID Услуг
    const SERVICE_VIP = 17; # приоритетное размещение объектов

    public function init()
    {
        parent::init();

        $this->module_title = 'Справочная';

        bff::autoloadEx(array(
            'ItemsImages' => array('app', 'modules/items/items.images.php'),
        ));
        # инициализируем модуль дин. свойств
        if (strpos(bff::$event, 'dynprops') === 0) {
            $this->dp();
        }
    }

    /**
     * Shortcut
     * @return Items
     */
    public static function i()
    {
        return bff::module('items');
    }

    /**
     * Shortcut
     * @return ItemsModel
     */
    public static function model()
    {
        return bff::model('items');
    }

    /**
     * Формирование URL
     * @param string $key ключ
     * @param mixed $opts параметры
     * @param boolean $dynamic динамическая ссылка
     * @return string
     */
    public static function url($key = '', $opts = '', $dynamic = false)
    {
        $base = static::urlBase(LNG, $dynamic);
        switch ($key)
        {
            # Просмотр
            case 'view':
                if (is_string($opts)) {
                    if ( ! $dynamic) {
                        return strtr($opts, array(
                            '{sitehost}' => SITEHOST . bff::locale()->getLanguageUrlPrefix(),
                        ));
                    } else {
                        return $opts;
                    }
                } else if (is_array($opts)) {
                    return static::url('view', $opts['link'], $dynamic).
                        ( ! empty($opts['tab']) && $opts['tab']!=='info' ? $opts['tab'].'/' : '' ).
                        ( ! empty($opts['q']) ? static::urlQuery($opts['q'], array('link', 'tab')) : '' );
                }
                break;
            # Форма добавления
            case 'add':
                return  $base.'/add.html'.static::urlQuery($opts);
                break;
            # Карта
            case 'map':
                return  $base.'/'.
                    (!empty($opts['full']) ? 'full/' : '').
                    (!empty($opts['cat']) ? $opts['cat'] : '').
                    static::urlQuery($opts, array('cat','full'));
                break;
            # Телефоны
            case 'phones':
                return  $base.'/phones/'.
                    (!empty($opts['cat']) ? $opts['cat'] : '').
                    static::urlQuery($opts, array('cat'));
                break;
        }
        return $base;
    }

    /**
     * URL объекта
     * @param integer $id ID объекта
     * @param string $keyword
     * @param int $cityID ID города или 0
     * @param bool $full
     * @return string
     */
    public static function urlView($id, $keyword = '', $cityID = 0, $full = true)
    {
        if (empty($keyword)) $keyword = 'item';
        return ( $full ? static::urlBase(LNG, true, array('city'=>$cityID)).'/' : '').$keyword.'-'.$id.'/';
    }

    /**
     * Описание seo шаблонов страниц
     * @return array
     */
    public function seoTemplates()
    {
        $dateMacros = array(
            'date' => array('t' => 'Дата ('.tpl::date_format2(978300000, false, true, ' ', ', ', true).')'),
            'date2' => array('t' => 'Дата ('.tpl::date_format2(978300000, false, false).')'),
            'date3' => array('t' => 'Дата ('.tpl::dateFormat(978300000).')'),
        );
        $aTemplates = array(
            'pages' => array(
                'map-index' => array( // listing
                    't'      => 'Карта - главная',
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'map-category' => array( // listing
                    't'      => 'Карта - категория',
                    'inherit'=> true,
                    'macros' => array(
                        'category' => array('t' => 'Название категории'),
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                        'seotext' => array(
                            't'    => 'SEO текст',
                            'type' => 'wy',
                        ),
                    ),
                ),
                'phones' => array( // phones
                    't'      => 'Телефоны - главная',
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'phones-category' => array( // phones
                    't'      => 'Телефоны - категория',
                    'list'   => true,
                    'inherit'=> true,
                    'macros' => array(
                        'category' => array('t' => 'Название категории'),
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'add' => array( // add
                    't'      => 'Добавление объекта',
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        // ...
                    ),
                ),
                'view' => array( // view
                    't'      => 'Просмотр объекта',
                    'inherit'=> true,
                    'macros' => array(
                        'tab'    => array('t' => 'Таб описания'),
                        'title'  => array('t' => 'Заголовок'),
                        'description'  => array('t' => 'Описание (до 150 символов)'),
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'share_title'       => array(
                            't'    => 'Заголовок (поделиться в соц. сетях)',
                            'type' => 'text',
                        ),
                        'share_description' => array(
                            't'    => 'Описание (поделиться в соц. сетях)',
                            'type' => 'textarea',
                        ),
                        'share_sitename'    => array(
                            't'    => 'Название сайта (поделиться в соц. сетях)',
                            'type' => 'text',
                        ),
                    ),
                ),
                'tv' => array( // services: tv
                    't'      => 'ТВ программа',
                    'macros' => array_merge(array(
                        'region' => array('t' => 'Регион'),
                    ), $dateMacros),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'tv-view' => array( // services: tv_view
                    't'      => 'ТВ программа: просмотр',
                    'macros' => array(
                        'title' => array('t' => 'Залоговок программа'),
                        'date' => array('t' => 'Дата и время показа'),
                        'channel' => array('t' => 'Название телеканала'),
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        // ...
                    ),
                ),
                'weather' => array( // services: weather
                    't'      => 'Погода',
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        // ...
                    ),
                ),
            ),
        );

        return $aTemplates;
    }

    /**
     * @return ItemsImages объект
     */
    public function initImages($nItemID = 0)
    {
        static $i;
        if (!isset($i)) {
            $i = new ItemsImages();
        }
        $i->setRecordID($nItemID);
        return $i;
    }

    /**
     * Максимально доступное кол-во изображений, прикрепляемых к объекту
     * @return integer
     */
    public static function imagesLimit()
    {
        return (int)config::sys('items.images.limit', 8);
    }

    /**
     * Максимально доступный размер файла изображения, прикрепляемый к объекту
     * @return integer по-умолчанию 5 Мб
     */
    public static function imagesMaxSize()
    {
        return (int)config::sys('items.images.maxsize', 5242880);
    }

    /**
     * Лимит категорий связанных с объектом, 0 - без ограничения
     * @return integer
     */
    public static function categoriesLimit()
    {
        return (int)config::sys('items.categories.limit', 0);
    }

    /**
     * Инициализация компонента работы с дин. свойствами
     * @return \bff\db\Dynprops объект
     */
    public function dp()
    {
        static $dynprops = null;
        if (isset($dynprops)) {
            return $dynprops;
        }

        # подключаем "Динамические свойства"
        $dynprops = new Dynprops('owner_id',
            TABLE_ITEMS_CATEGORIES,
            TABLE_ITEMS_CATEGORIES_DP,
            TABLE_ITEMS_CATEGORIES_DPM,
            1); # полное наследование
        $this->attachComponent('dynprops', $dynprops);

        $dynprops->setSettings(array(
                'module_name'          => $this->module_name,
                'typesAllowed'         => array(
                    Dynprops::typeCheckboxGroup,
                    Dynprops::typeRadioGroup,
                    Dynprops::typeRadioYesNo,
                    Dynprops::typeCheckbox,
                    Dynprops::typeSelect,
                    Dynprops::typeInputText,
                    Dynprops::typeTextarea,
                    Dynprops::typeNumber,
                    Dynprops::typeRange,
                ),
                'ownerTableType'       => 2,
                'langs'                => $this->locale->getLanguages(false),
                'langText'             => array(
                    'yes'    => _t('', 'Да'),
                    'no'     => _t('', 'Нет'),
                    'all'    => _t('', 'Все'),
                    'select' => _t('', 'Выбрать'),
                ),
                'cache_method'         => 'Items_dpSettingsChanged',
                'typesAllowedParent'   => array(Dynprops::typeSelect),
                /**
                 * Настройки доступных int/text столбцов динамических свойств для хранения числовых/тестовых данных.
                 * При изменении, не забыть привести в соответствие столбцы f(1-n) в таблице TABLE_ITEMS
                 */
                'datafield_int_last'   => 15,
                'datafield_text_first' => 16,
                'datafield_text_last'  => 20,
                'searchRanges'         => true,
                'cacheKey'             => false,
            )
        );

        return $dynprops;
    }

    /**
     * Получаем дин. свойства категории
     * @param integer $nCategoryID ID категории
     * @param boolean $bResetCache обнулить кеш
     * @return mixed
     */
    public function dpSettings($nCategoryID, $bResetCache = false)
    {
        if ($nCategoryID <= 0) {
            return array();
        }

        $cache = Cache::singleton('items-dp', 'file');
        $cacheKey = 'cat-dynprops-' . LNG . '-' . $nCategoryID;
        if ($bResetCache) {
            # сбрасываем кеш настроек дин. свойств категории
            return $cache->delete($cacheKey);
        } else {
            if (($aSettings = $cache->get($cacheKey)) === false) { # ищем в кеше
                $aSettings = $this->dp()->getByOwner($nCategoryID, true, true, false);
                $cache->set($cacheKey, $aSettings); # сохраняем в кеш
            }

            return $aSettings;
        }
    }

    /**
     * Метод вызываемый модулем \bff\db\Dynprops, в момент изменения настроек дин. свойств категории
     * @param integer $nCategoryID id категории
     * @param integer $nDynpropID id дин.свойства
     * @param string $sEvent событие, генерирующее вызов метода
     * @return mixed
     */
    public function dpSettingsChanged($nCategoryID, $nDynpropID, $sEvent)
    {
        if (empty($nCategoryID)) {
            return false;
        }
        $this->dpSettings($nCategoryID, true);
    }

    /**
     * Формирование SQL запроса для сохранения дин.свойств
     * @param integer $nCategoryID ID подкатегории
     * @param string $sFieldname ключ в $_POST массиве
     * @return array
     */
    public function dpSave($nCategoryID, $sFieldname = 'd')
    {
        $aData = $this->input->post($sFieldname, TYPE_ARRAY);

        $aDynpropsData = array();
        foreach ($aData as $props) {
            foreach ($props as $id => $v) {
                $aDynpropsData[$id] = $v;
            }
        }

        $aDynprops = $this->dp()->getByID(array_keys($aDynpropsData), true);

        return $this->dp()->prepareSaveDataByID($aDynpropsData, $aDynprops, 'update', true);
    }

    /**
     * Формирование формы редактирования/фильтра дин.свойств
     * @param integer $nCategoryID ID категории
     * @param boolean $bSearch формирование формы поиска
     * @param array|boolean $aData данные или FALSE
     * @param array $aExtra доп.данные
     * @return string HTML template
     */
    protected function dpForm($nCategoryID, $bSearch = true, $aData = array(), $aExtra = array())
    {
        if (empty($nCategoryID)) {
            return '';
        }

        if (bff::adminPanel()) {
            if ($bSearch) {
                $aForm = $this->dp()->form($nCategoryID, $aData, true, true, 'd', 'search.inline', false, $aExtra);
            } else {
                $aForm = $this->dp()->form($nCategoryID, $aData, true, false, 'd', 'form.table', false, $aExtra);
            }
        } else {
            if (!$bSearch) {
                $aForm = $this->dp()->form($nCategoryID, $aData, true, false, 'd', 'item.form.dp', $this->module_dir_tpl, $aExtra);
            }
        }

        return (!empty($aForm['form']) ? $aForm['form'] : '');
    }

    /**
     * Отображение дин. свойств
     * @param integer $nCategoryID ID категории
     * @param array $aData данные
     */
    public function dpView($nCategoryID, $aData)
    {
        $sKey = 'd';
        if (!bff::adminPanel()) {
            $aForm = $this->dp()->form($nCategoryID, $aData, true, false, $sKey, 'view.dp', $this->module_dir_tpl);
        } else {
            $aForm = $this->dp()->form($nCategoryID, $aData, true, false, $sKey, 'view.table');
        }

        return (!empty($aForm['form']) ? $aForm['form'] : '');
    }

    /**
     * Подготовка запроса полей дин. свойств на основе значений "cache_key"
     * @param string $sPrefix префикс таблицы, например "I."
     * @param int $nCategoryID ID категории
     * @return string
     */
    public function dpPrepareSelectFieldsQuery($sPrefix = '', $nCategoryID = 0)
    {
        if (empty($nCategoryID) || $nCategoryID<0) {
            return '';
        }

        $fields = array();
        foreach($this->dpSettings($nCategoryID) as $v)
        {
            $f = $sPrefix.$this->dp()->datafield_prefix.$v['data_field'];
            if(!empty($v['cache_key'])) {
               $f .= ' as `'.$v['cache_key'].'`';
            }
             $fields[] = $f;
        }
        return (!empty($fields) ? join(', ', $fields) : '');
    }

    /**
    * @param integer $nParentID ID выбранной категории
    * @param bool $bIncludingSubcategories выключая подкатегории
    * @param bool $mAddRootCategory включая корневую категорию
    * @param integer $nType 0 - все, 1 - только категории без подкатегорий, 2 - только категории без объектов
    * @param bool $bOnlyEnabled только включенные категории
    * @param array $aIgnoreCategoriesID ID игнорируемых категорий
    * @return string select::options
    */
    function categoryOptions($nParentID=null, $bIncludingSubcategories=true, $mAddRootCategory=false, $nType=0, $bOnlyEnabled=false, $aIgnoreCategoriesID = array())
    {
        $sDisabledStyle = 'style="color: #999;"';
        $sSelectPart = 'IC.id, ICL.title, IC.pid, IC.enabled';
        $aIgnoreCategoriesID = $this->db->prepareIN('IC.id', $aIgnoreCategoriesID, true);
        //&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;
        
        //достаем категории из базы и формируем options
        $sParentOptions ='';
        if ($bIncludingSubcategories)
        {
            if ($nType==2)
            {
                $aCategories = $this->db->transformRowsToTree( $this->db->select('SELECT '.$sSelectPart.', I.item_id as items
                                         FROM '.TABLE_ITEMS_CATEGORIES.' IC
                                            LEFT JOIN '.TABLE_ITEMS_IN_CATEGORIES.' I ON I.category_id = IC.id,
                                            '.TABLE_ITEMS_CATEGORIES_LANG.' ICL
                                         WHERE '.$aIgnoreCategoriesID.($bOnlyEnabled?' IC.enabled=1 ':'').
                                                $this->db->langAnd(true, 'IC', 'ICL').'
                                         GROUP BY IC.id
                                         ORDER BY IC.num'), 'id', 'pid', 'subitems');
            } else {
                $aCategories = $this->db->transformRowsToTree( $this->db->select('SELECT '.$sSelectPart.'
                                         FROM '.TABLE_ITEMS_CATEGORIES.' IC,
                                              '.TABLE_ITEMS_CATEGORIES_LANG.' ICL
                                         WHERE '.$aIgnoreCategoriesID.($bOnlyEnabled?' WHERE IC.enabled=1 ':'').
                                                $this->db->langAnd(true, 'IC', 'ICL').'
                                         ORDER BY IC.num'), 'id', 'pid', 'subitems');
            }
            $bOptgroup = false;
            foreach ($aCategories as $cat)
            {            
                $bOptgroup = (($nType == 1 && !empty($cat['subitems'])) || ($nType == 2 && $cat['items']));
                if ($bOptgroup)
                {
                    $sParentOptions .= '<optgroup label="'.$cat['title'].'">';
                } else {
                    $sParentOptions .= '<option value="'.$cat['id'].'" pid="'.$cat['pid'].'"'.($nParentID == $cat['id']?' selected ':'').(!$cat['enabled']?$sDisabledStyle:'').'>'.$cat['title'].'</option>';
                }
                
                if (!empty($cat['subitems'])) {
                    foreach ($cat['subitems'] as $subcat) {
                        if ($nType == 2 && $subcat['items']) continue;
                        $sParentOptions .= '<option style="padding-left:20px;" '.(!$subcat['enabled']?$sDisabledStyle:'').' value="'.$subcat['id'].'" pid="'.$subcat['pid'].'"'.($nParentID == $subcat['id']?' selected ':'').'>'.$subcat['title'].'</option>';
                    }
                }

                if ($bOptgroup) {
                    $sParentOptions .= '</optgroup>';
                }
            }
        }
        else
        {
            $aCategories = $this->db->select('
                SELECT '.$sSelectPart.'
                FROM '.TABLE_ITEMS_CATEGORIES.' IC,
                     '.TABLE_ITEMS_CATEGORIES_LANG.' ICL
                WHERE IC.pid=0  AND '.$aIgnoreCategoriesID.
                     $this->db->langAnd(true, 'IC', 'ICL').'
                ORDER BY IC.num');
            foreach ($aCategories as $cat)
                $sParentOptions .= '<option value="'.$cat['id'].'" pid="'.$cat['pid'].'"'.($nParentID == $cat['id']?' selected ':'').(!$cat['enabled']?$sDisabledStyle:'').'>'.$cat['title'].'</option>';
        }
        
        if ($mAddRootCategory)
             $sParentOptions = '<option style="font-weight:bold;" value="0">'.($mAddRootCategory === TRUE ? 'корневая категория' : $mAddRootCategory).'</option>'.$sParentOptions;
        
        return $sParentOptions;
    }
    
    function categoryIconUpdate($nCategoryID, $bDeletePrevious=false, $bDoUpdateQuery = false, $sFieldName = 'icon')
    {
        if ($nCategoryID && !empty($_FILES) && !empty($_FILES[$sFieldName]['name']))
        {
            if ($bDeletePrevious)
                $this->categoryIconDelete($nCategoryID);

            $aImageSize = getimagesize($_FILES[$sFieldName]['tmp_name']);
            if ($aImageSize!==FALSE && in_array($aImageSize[2], array(IMAGETYPE_GIF, IMAGETYPE_JPEG, IMAGETYPE_PNG)))
            {   
                $sExtension = image_type_to_extension($aImageSize[2], true); 
                $sFilename  = $nCategoryID.$sExtension;
                                                                                                         
                # 25x25
                $th = new Thumbnail($_FILES[$sFieldName]['tmp_name'], false);
                $th->save(array(array(
                    'width'=>25,'height'=>25,
                    'filename'=>bff::path('icategories', 'images').$sFilename,
                )));

                if ($bDoUpdateQuery) {
                    $this->db->exec('UPDATE '.TABLE_ITEMS_CATEGORIES.' SET icon='.$this->db->str2sql($sFilename).' WHERE id='.$nCategoryID);
                }
                
                return $sFilename;
            } 
        }
        return false;
    }

    function categoryIconDelete($mCategoryID, $bDoUpdateQuery = false)
    {
        if (!empty($mCategoryID))
        {
            if (is_array($mCategoryID))
            {
                $mCategoryID = array_map('intval', $mCategoryID);
                $aFilenames = $this->db->select_one_column('SELECT icon FROM '.TABLE_ITEMS_CATEGORIES.' WHERE id IN ('.implode(',', $mCategoryID).') ');
                if (!empty($aFilenames))
                {
                    foreach ($aFilenames as $file) {
                        if (!empty($file))
                            @unlink(bff::path('icategories', 'images').$file);
                    } 
                    if ($bDoUpdateQuery) {
                        $this->db->exec('UPDATE '.TABLE_ITEMS_CATEGORIES.' SET icon='.$this->db->str2sql('').' WHERE id IN ('.implode(',', $mCategoryID).') ');
                    }
                    return true;
                }
            } else {
                $mCategoryID = intval($mCategoryID);
                
                $sFilename = $this->db->one_data('SELECT icon FROM '.TABLE_ITEMS_CATEGORIES.' WHERE id='.$mCategoryID);
                if (!empty($sFilename)) {
                    @unlink(bff::path('icategories', 'images').$sFilename);
                    if ($bDoUpdateQuery) {
                        $this->db->exec('UPDATE '.TABLE_ITEMS_CATEGORIES.' SET icon='.$this->db->str2sql('').' WHERE id = '.$mCategoryID.' ');
                    }
                    return true;
                }
            }
        }
        return false;
    }

    function itemDelete($nItemID)
    {
        if (empty($nItemID))
            return false;

        $aData = $this->model->itemData($nItemID, array('id','user_id','agent_id','moderated'));
        if (empty($aData)) return false;
        
        # удаляем объект
        $res = $this->db->delete(TABLE_ITEMS, array('id'=>$nItemID));
        if (empty($res)) return false;

        $this->db->delete(TABLE_ITEMS_LANG, array('id'=>$nItemID));
        
        # удаляем изображения
        $oImages = $this->initImages($nItemID);
        $oImages->deleteAllImages(false);

        # удаляем привязку к категориям
        $this->db->delete(TABLE_ITEMS_IN_CATEGORIES, array('item_id'=>$nItemID));
        
        # удаляем адреса
        $this->db->delete(TABLE_ITEMS_ADDR, array('item_id'=>$nItemID));
       
        # удаляем отзывы
        $nUnmoderated = $this->db->one_data('SELECT COUNT(id) FROM '.TABLE_ITEMS_OPINIONS.'
                WHERE item_id = :id AND moderated = 0 GROUP BY id', array(':id'=>$nItemID));
        if ($nUnmoderated > 0) {
            $this->itemsOpinionsModerationCounter( false, $nUnmoderated );
        }
        $this->db->delete(TABLE_ITEMS_OPINIONS, array('item_id'=>$nItemID));

        # удаляем "режим работы"
        $this->db->delete(TABLE_ITEMS_SCHEDULE, array('item_id'=>$nItemID));
        $this->db->delete(TABLE_ITEMS_SCHEDULE_OFF, array('item_id'=>$nItemID));

        # счетчик заявок на представительство
        $nRequests = $this->db->select_rows_count(TABLE_ITEMS_AGENTS_REQUESTS, array('item_id'=>$nItemID));
        if ($aData['agent_id'] > 0) {
            # откручиваем счетчик объектов пользователя (которые он представляет)
            $this->security->userCounter('items', -1, $aData['agent_id']);
        } else if ($aData['user_id'] > 0 && $nRequests > 0) {
            # откручиваем счетчик заявок пользователя на "представительство"
            $this->security->userCounter('items_requests', -1, $aData['user_id']);
        }

        # счетчик заявок на представительство
        if ($nRequests > 0) {
            $this->db->delete(TABLE_ITEMS_AGENTS_REQUESTS, array('item_id'=>$nItemID));
            config::saveCount('items_agents', -$nRequests, true);
        }
        
        return true;
    }

    function renumItemCategory($nItemID, $aCategoryID)
    {
        if ($nItemID>0 && !empty($aCategoryID))
        {   
            $aUpdateData = array(); $i=1;
            foreach ($aCategoryID as $v){
                $aUpdateData[] = "WHEN $v THEN $i"; $i++;
            }
            if (!empty($aUpdateData)) {
                $this->db->exec('UPDATE '.TABLE_ITEMS_IN_CATEGORIES.'
                                    SET num = CASE category_id '.join(' ', $aUpdateData).' ELSE num END 
                                    WHERE item_id = '.$nItemID );
            }
        }
    }
              
    function saveItemAddr($nItemID = 0, array $aAddr, $bNewItem = false)
    {
        if (!empty($aAddr) && is_array($aAddr) && $nItemID>0)
        {
            # чистим некоторые поля
            $this->input->clean_array($aAddr, array(
                'id'          => TYPE_UINT,
                'lat'         => TYPE_NUM,
                'lng'         => TYPE_NUM,
                'city_id'     => TYPE_UINT,
                'phones'      => array(
                    TYPE_ARRAY_ARRAY,
                    'v' => TYPE_NOTAGS,
                    't' => TYPE_NOTAGS,
                ),
                'emails'      => array(
                    TYPE_ARRAY_ARRAY,
                    'v' => TYPE_NOTAGS,
                    't' => TYPE_NOTAGS,
                ),
                'sites'      => array(
                    TYPE_ARRAY_ARRAY,
                    'v' => TYPE_NOTAGS,
                    't' => TYPE_NOTAGS,
                ),
                'description' => TYPE_ARRAY,
                'title'       => TYPE_STR,
            ));

            # телефоны
            $aAddr['phonesq'] = array();
            if (empty($aAddr['phones'])) $aAddr['phones'] = array();
            foreach ($aAddr['phones'] as $phoneKey=>&$phoneData)
            {
                if (empty($phoneData['v'])) { # если пустой телефон, удаляем
                    unset($aAddr['phones'][$phoneKey]);
                    continue;
                }

                # чистим телефон
                $phoneData['v'] = preg_replace('/[^\(\)\+\-0-9 ]/','',trim($phoneData['v']));
                $phoneData['v'] = preg_replace("/\s+/", ' ', $phoneData['v']); //сжимаем двойные пробелы
                if (empty($phoneData['v'])) { // если пустой телефон, удаляем
                    unset($aAddr['phones'][$phoneKey]);
                    continue;
                }

                if (sizeof($aAddr['phonesq'])<2) {
                    $aAddr['phonesq'][] = $phoneData['v'];
                }

                # чистим название телефона
                $phoneData['t'] = mb_substr( trim(strip_tags($phoneData['t'])), 0, 100 );
            }

            # почта
            if (empty($aAddr['emails'])) $aAddr['emails'] = array();
            foreach ($aAddr['emails'] as $emailKey=>&$emailData)
            {
                if (empty($emailData['v'])) { # если пустой email, удаляем
                    unset($aAddr['emails'][$emailKey]);
                    continue;
                }

                # чистим email
                $emailData['v'] = mb_substr( trim(strip_tags($emailData['v'])), 0, 100 );
                if (empty($emailData['v'])) { # если пустой email, удаляем
                    unset($aAddr['emails'][$emailKey]);
                    continue;
                }

                # чистим название email'a
                $emailData['t'] = mb_substr( trim(strip_tags($emailData['t'])), 0, 100 );
            }

            # WWW
            $aAddr['site'] = '';
            if (empty($aAddr['sites'])) $aAddr['sites'] = array();
            foreach ($aAddr['sites'] as $siteKey=>&$siteData)
            {
                if (empty($siteData['v'])) { // если пустой сайт, удаляем
                    unset($aAddr['sites'][$siteKey]);
                    continue;
                }

                # чистим сайт
                $siteData['v'] = mb_substr( trim(strip_tags($siteData['v'])), 0, 255 );
                $siteData['v'] = str_replace(array('http://','https://', 'ftp://'), '', $siteData['v']);
                if (empty($siteData['v'])) { // если пустой сайт, удаляем
                    unset($aAddr['sites'][$siteKey]);
                    continue;
                } if (empty($aAddr['site'])){
                    $aAddr['site'] = $siteData['v'];
                }

                # чистим название сайт'a
                $siteData['t'] = mb_substr( trim(strip_tags($siteData['t'])), 0, 200 );
            }

            foreach ($aAddr['description'] as $lng => $v){
                $this->input->clean($aAddr['description'][$lng], TYPE_NOTAGS);
                if ($v == '<br/>') $aAddr['description'][$lng] = '';
            }


            if ($aAddr['id']>0) {
                $aUpdate = array(
                    'lat' => $aAddr['lat'], 'lng' => $aAddr['lng'],
                    'city_id' => $aAddr['city_id'],
                    'address' => $aAddr['address'],
                    'description' => $aAddr['description'],
                    'phones' => serialize($aAddr['phones']),
                    'phonesq' => (!empty($aAddr['phonesq']) ? join(', ', $aAddr['phonesq']) : null),
                    'emails' => serialize($aAddr['emails']),
                    'sites' => serialize($aAddr['sites']),
                    'site' => $aAddr['site'],
                );
                $this->db->langFieldsModify($aUpdate, $this->model->langItemsAddr, $aUpdate);

                $this->db->update(TABLE_ITEMS_ADDR, $aUpdate, array('id'=>$aAddr['id'], 'item_id'=>$nItemID));
            } else {
                $aInsert = array(
                    'item_id'=>$nItemID,
                    'lat'=>$aAddr['lat'], 'lng'=>$aAddr['lng'],
                    'city_id'=>$aAddr['city_id'], 'address'=>$aAddr['address'],
                    'description'=>$aAddr['description'],
                    'phones'=>serialize($aAddr['phones']),
                    'phonesq'=>join(',', $aAddr['phonesq']),
                    'emails'=>serialize($aAddr['emails']),
                    'sites'=>serialize($aAddr['sites']),
                    'site' => $aAddr['site'],
                );
                $this->db->langFieldsModify($aInsert, $this->model->langItemsAddr, $aInsert);
                $this->db->insert(TABLE_ITEMS_ADDR, $aInsert);
            }
            return $aAddr;
        }
        return false;
    }

    function prepareItemAddrContactsSave(&$aData)
    {
        # телефоны
        $aTemp = array(); $limit = 3;
        foreach ($aData['phones'] as $p) {
            $p = preg_replace('/[^\(\)\+\-0-9 ]/','',trim($p));
            $p = preg_replace('/\s+/', ' ', $p);
            if (strlen($p)>4) { $aTemp[] = array('v'=>mb_substr($p, 0, 40), 't'=>''); }
        }
        if ($limit > 0 && sizeof($aTemp) > $limit) $aTemp = array_slice($aTemp, 0, $limit);
        if (empty($aTemp)) { # минимум 1 телефон
            $this->errors->set( _t('items', 'Укажите телефон фирмы') ); 
        }
        $aData['phones'] = serialize($aTemp);
        
        # сайты
        $aTemp = array(); $limit = 3;
        foreach ($aData['sites'] as $s) {
            $s = mb_substr( trim(strip_tags($s)), 0, 255 ); 
            if (strlen($s)>3) { $aTemp[] = array('v'=>strtr($s, array('http://'=>'','https://'=>'')), 't'=>''); }
        }
        if ($limit > 0 && sizeof($aTemp) > $limit) $aTemp = array_slice($aTemp, 0, $limit);
        $aData['sites'] = serialize($aTemp);
        
        # email'ы
        $aTemp = array(); $limit = 3;
        foreach ($aData['emails'] as $e) {
            if ($this->input->isEmail($e)) { $aTemp[] = array('v'=>$e, 't'=>''); }
        }
        if ($limit > 0 && sizeof($aTemp) > $limit) $aTemp = array_slice($aTemp, 0, $limit);
        $aData['emails'] = serialize($aTemp);      
    }
    
    # Расписание работы
    function saveItemSchedule($nItemID, $aSchedule, $aScheduleOff)
    {
        if (!$nItemID) return false;
        if (empty($aSchedule) && empty($aScheduleOff)) return false;
        
        # Рабочие дни
        $aHtml = array('work'=>array(), 'off'=>array());
        $sqlInsert = array();
        $sqlDelete = array();
        $nDays = 0;
        foreach ($aSchedule as $v)
        {   
            $this->input->clean_array($v, array(
                'id'    => TYPE_UINT,
                'days'  => TYPE_ARRAY_UINT,
                'del'   => TYPE_BOOL,
                'time_from' => TYPE_UINT, 
                'time_to'   => TYPE_UINT, 
                'time_full' => TYPE_BOOL,
                'break_from'=> TYPE_UINT, 
                'break_to'  => TYPE_UINT,
                'no_break'  => TYPE_BOOL,
            ));
            $id = $v['id']; unset($v['id']);

            # дни
            $nDaysInput = array_sum($v['days']); $v['days'] = 0;
            for ($i=1;$i<=64;$i*=2) {
                if ($nDaysInput & $i && !($nDays&$i)) {
                    $v['days'] += $i; $nDays += $i;
                }
            }
            if ($v['days'] == 0) {
                if ($id) {
                    $sqlDelete[] = $id;
                }
                unset($v);
                continue;
            }
            if (!empty($v['del']) && $id) {
                $sqlDelete[] = $id;
                unset($v); continue;
            }
            
            # время работы
            $this->input->clean($v['time_from'], TYPE_UINT);
            if ($v['time_from']>1410) $v['time_from'] = 0;
            $this->input->clean($v['time_to'], TYPE_UINT);
            if ($v['time_to']>1440) $v['time_to'] = 0;
            if ($v['time_full'] || ($v['time_from'] == 0 && $v['time_to'] == 1440)) {
                $v['time_full'] = 1;
                $v['time_from'] = 0;
                $v['time_to'] = 1440;
            } else {
                $v['time_full'] = 0;
            }

            # перерыв
            if (!empty($v['no_break'])) {
                $v['no_break'] = 1;
                $v['break_from'] = $v['break_to'] = 0;             
            } else {
                $v['no_break'] = 0;
                $this->input->clean($v['break_from'], TYPE_UINT);
                if ($v['break_from']>1410) $v['break_from'] = 0;
                $this->input->clean($v['break_to'], TYPE_UINT);
                if ($v['break_to']>1440) $v['break_to'] = 0;
                # если перерыв "до" <= "с", значит будет "c"+1 час
                if ($v['break_to']<=$v['break_from']) $v['break_to'] = $v['break_from']+60;
            }
            
            $aHtml['work'][] = $this->getItemScheduleHtml($v);
            
            if ($id)
            {   unset($v['del']);
                $this->db->update(TABLE_ITEMS_SCHEDULE, $v, array('id'=>$id, 'item_id'=>$nItemID));
            } else { 
                unset($v['del']);
                $sqlInsert[] = array_merge($v, array('item_id'=>$nItemID));
            }
        }
        
        if ( ! empty($sqlInsert)) {
            $this->db->multiInsert(TABLE_ITEMS_SCHEDULE, $sqlInsert);
        }
        if ( ! empty($sqlDelete)) {
            $this->db->delete(TABLE_ITEMS_SCHEDULE, array('item_id'=>$nItemID, 'id'=>$sqlDelete));
        }

        # Выходные дни
        $noOffDays = true;
        $sqlInsert = array();
        $sqlDelete = array();
        foreach ($aScheduleOff as $v)
        {   
            $this->input->clean_array($v, array(
                'id'   => TYPE_UINT,
                'days' => TYPE_ARRAY_UINT,
                'del'  => TYPE_BOOL,
                'description' => TYPE_STR,
            ));
            $id = $v['id']; unset($v['id']);
            
            # дни
            $nDaysInput = array_sum($v['days']); $v['days'] = 0;
            for ($i=1;$i<=64;$i*=2) {
                if ($nDaysInput & $i) {
                    $v['days'] += $i;
                }
            }
            if ($v['del'] == 1 && $id) {
                $sqlDelete[] = $id;
                unset($v); continue;
            }
            
            # примечание
            $v['description'] = ( ! empty($v['description']) ? mb_substr( $this-> input->clean($v['description'], TYPE_STR), 0, 150) : '');
            
            $aHtml['off'][] = $this->getItemScheduleHtml($v, true); 
            if ($v['days'] > 0) $noOffDays = false;
            
            if ($id)
            {   unset($v['del']);
                //update query
                $this->db->update(TABLE_ITEMS_SCHEDULE_OFF, $v, array('id'=>$id, 'item_id'=>$nItemID));
            } else { 
                //insert query
                $sqlInsert[] = array('item_id'=>$nItemID, 'days'=>$v['days'], 'description'=>$v['description']);
            }
        }
        
        if (!empty($sqlInsert)) {
            $this->db->multiInsert(TABLE_ITEMS_SCHEDULE_OFF, $sqlInsert);
        }
        if (!empty($sqlDelete)) {
            $this->db->delete(TABLE_ITEMS_SCHEDULE_OFF, array('item_id'=>$nItemID, 'id'=>$sqlDelete));
        }

        return $this->db->update(TABLE_ITEMS, array(
            'schedule_work' => join('<br/>', $aHtml['work']),
            'schedule_off'  => (!empty($aHtml['off']) ? '<br/>' : ''). ($noOffDays ? '<i>без выходных</i><br/>' : '') . join('<br/>', $aHtml['off']),
        ), array('id'=>$nItemID));
    }
    
    function getItemScheduleHtml($v, $bOff = false)
    {
        $html = '';
        if (!$bOff)
        {
            $html .= '<b>'.$this->getItemScheduleViewDays($v['days'], $cnt).'</b>';
            $html .= '<br/>';
            if ($v['time_full'])
            {
                $html .= '<i>'._t('items', 'круглосуточно').'</i>';
            } else {
                $html .= _t('','c').' '.$this->getItemScheduleViewTime($v['time_from']).' '._t('','до').' '.$this->getItemScheduleViewTime($v['time_to']);
                if ($v['no_break'])
                {
                    $html .= '<br/><i>'._t('items', 'без перерыва').'</i>';
                } else {
                    $html .= '<br/><i>'._t('items', 'перерыв:').' '.$this->getItemScheduleViewTime($v['break_from']).' - '.$this->getItemScheduleViewTime($v['break_to']).'</i>';
                }                
            }
        } else {
            if ($v['days']>0) {
                $html .= '<b>'.$this->getItemScheduleViewDays($v['days'], $cnt).'</b>';
                $html .= '<br/><i>'.($cnt == 1 ? _t('items', 'выходной день') : _t('items', 'выходные дни')).'</i>';
            }                
            if (!empty($v['description'])) {
                if ($v['days']>0) $html .= '<br/>';
                $html .= '<i>'.nl2br($v['description']).'</i>';
            }
        }
        return $html.'<br/>';
    }
    
    function getItemScheduleViewTime($nTime, $bSkip00Minutes = false)
    {
        if (!$nTime) return '00'.(!$bSkip00Minutes?':00':'');
        $res = ($nTime/30)/2;
        $hour = intval($res);
        return $hour.($res>$hour?':30':(!$bSkip00Minutes?':00':''));
    }

    function getItemScheduleViewDays($nDays, &$cnt=0)
    {
        if (!$nDays) return '';
        
        $aResult = array();
        $sDays = array(1 => _t('','понедельник'), 2 => _t('','вторник'), 4 => _t('','среда'), 8 => _t('','четверг'), 16 => _t('','пятница'), 32 => _t('','суббота'), 64 => _t('','воскресенье'));
        for ($i=0,$v=1,$j=0,$s=1;$v<=64;$v*=2,$i++) {
            if ($nDays & $v) {
                if (!$j) $s=$v;
                $j++;           
            } else {
                if ($j>0) {
                    if ($j>2) {
                        $aResult[] = $sDays[$s].'-'.$sDays[$v/2];
                    } 
                    else if ($j == 2) {
                        $aResult[] = join(', ', array_slice($sDays, $i-$j, $j, true));
                        $cnt += ($j - $i-$j);
                    } else {
                       $aResult[] = $sDays[$v/2];
                       $cnt++;
                    }
                }
                $j = 0; 
            }
        }
        if ($j>0) {
            if ($j>2) {
                $aResult[] = $sDays[$s].'-'.$sDays[$v/2];
            } 
            else if ($j == 2) {
                $aResult[] = join(', ', array_slice($sDays, $i-$j, $j, true));
                $cnt += ($j - $i-$j);
            } else {
               $aResult[] = $sDays[$v/2];
               $cnt++;
            }
        } 
                     
        return join(', ', $aResult);
    }
    
    /**
     * Проверка существования ключа
     * @param string $sKeyword ключ
     * @param string $sTable таблица
     * @param integer $nExceptRecordID исключая записи (ID)
     * @returns integer
     */ 
    function isKeywordExists($sKeyword, $sTable, $nExceptRecordID = null)
    {
        return $this->db->one_data('SELECT id
                               FROM '.$sTable.'
                               WHERE '.( ! empty($nExceptRecordID)? ' id!='.intval($nExceptRecordID).' AND ' : '' ).'
                                  keyword='.$this->db->str2sql($sKeyword).'  LIMIT 1');
    }
    
    /**
     * Проверка ключа / Формирование ключа исходя из заголовка
     * @param string $sKeyword ключ
     * @param string $sTitle заголовок
     * @returns string ключ
     */ 
    function getKeyword($sKeyword = '', $sTitle = '')
    {                                            
        if (empty($sKeyword) && !empty($sTitle)) {
            $sKeyword = mb_strtolower( func::translit( $sTitle ) );
        }
        
        $sKeyword = preg_replace('/[^a-zA-Z0-9_\-\']/','',$sKeyword);
        if (empty($sKeyword)){
             $this->errors->set( _t('items', 'Укажите keyword') );
        } else {

        }
        
        return $sKeyword;              
    }
    
    function itemsModerationCounter($nIncrement = false)
    {
        $sCounterKey = 'items_mod';
        if ($nIncrement!==false) {
            config::saveCount($sCounterKey, $nIncrement, true);
        } else {
            return (int)config::get($sCounterKey, 0);
        }
    }

    function getItemClaimReasons()
    {
        return array(
            1  => _t('items', 'Неверная контактная информация'),
            2  => _t('items', 'Объект не соответствует рубрике'),
            8  => _t('items', 'Некорректная фотография'),
            32 => _t('items', 'Другое'),
        );        
    }
    
    function getItemClaimText($nReasons, $sComment)
    {
        $nOtherReason = 32;
        
        $reasons = $this->getItemClaimReasons();
        if (!empty($nReasons) && !empty($reasons))
        {
            $r_text = array();
            foreach ($reasons as $rk=>$rv) {
                if ($rk!=$nOtherReason && $rk & $nReasons) {
                    $r_text[] = $rv;
                }
            }
            $r_text = join(', ', $r_text);
            if ($nReasons & $nOtherReason && !empty($sComment)) {
                $r_text .= ', '.$sComment;
            }
            return $r_text;
        }
        return '';
    }

    public function itemsClaimsCounter($nIncrement = false)
    {
        $sCounterKey = 'items_claims';
        if ($nIncrement!==false) {
            config::saveCount($sCounterKey, $nIncrement, true);
        } else {
            return (int)config::get($sCounterKey, 0);
        }
    }
    
    function itemsOpinionsModerationCounter($bIncrement = false, $nCount = 1)
    {
        config::saveCount('items_opinions', ($bIncrement ? $nCount : -$nCount), true);
    }
    
    function itemOpinionsCacheUpdate($nItemID) 
    {
        // обновляем счетчики кол-ва отзывов
        $cnt = $this->db->one_array('SELECT COUNT(O.id) as total,
                            SUM(O.status = '.self::OPINION_STATUS_POSITIVE.') as plus,
                            SUM(O.status = '.self::OPINION_STATUS_NEUTRAL.') as neutral,
                            SUM(O.status = '.self::OPINION_STATUS_NEGATIVE.') as minus
                        FROM '.TABLE_ITEMS_OPINIONS.' O
                        WHERE O.item_id = '.$nItemID.' AND O.moderated = 1
                        GROUP BY O.item_id
                    ');
        if (empty($cnt)) {
            $cnt = array('total'=>0, 'plus'=>0, 'minus'=>0, 'neutral'=>0);
        }

        $sOpinionsCache = $cnt['plus'].','.$cnt['neutral'].','.$cnt['minus'];

        // сохраняем изменения
        $this->db->exec('UPDATE '.TABLE_ITEMS.'
            SET opinions_cache = '.$this->db->str2sql($sOpinionsCache).',
                opinions = '.intval($cnt['total']).'
            WHERE id = '.$nItemID);
    }

    # --------------------------------------------------------
    # Активация услуг

    public function svcActivate($nItemID, $nSvcID, $aSvcData = false, array &$aSvcSettings = array())
    {
        $svc = $this->svc();
        if ( ! $nSvcID ) {
            $this->errors->set(_t('svc', 'Неудалось активировать услугу'));
            return false;
        }
        if ( empty($aSvcData) || ! is_array($aSvcData) ) {
            $aSvcData = $svc->model->svcData($nSvcID);
            if ( empty($aSvcData) ) {
                $this->errors->set(_t('svc', 'Неудалось активировать услугу'));
                return false;
            }
        }

        switch ($nSvcID)
        {
            # приоритетное размещение объекта
            # срок действия услуги: 1 год
            case self::SERVICE_VIP:
            {
                $aItem = $this->db->one_array('SELECT id, vip, vip_expire FROM '.TABLE_ITEMS.' WHERE id = '.$nItemID);
                if (empty($aItem)) {
                    bff::log('Ошибка активации услуги "Приоритетное размещение" для несуществующего объекта #'.$nItemID, 'bills.log');
                    return false;
                }

                $now = time();
                $expire = strtotime($aItem['vip_expire']);
                if (empty($aItem['vip']) || $now >= $expire )
                {
                    // активация услуги (на текущий момент неактивированной, либо только что закончившей период действия)
                    $from = $now;
                    $to = strtotime('+1 year', $from); // + 1 год от текущей даты
                } else {
                    // продление активной услуги
                    $from = $expire;
                    $to = strtotime('+1 year', $from); // + 1 год от текущей даты завешения действия услуги
                }

                // активируем / продлеваем услугу
                $res = $this->db->update(TABLE_ITEMS,
                        array('vip_expire'=>date('Y-m-d H:i:s', $to),
                              'vip'=>1),
                        array('id'=>$nItemID));

                if (empty($res)) return false;

                return _t('','Приоритетное размещение').' '.tpl::date_format2( $from ).' - '.tpl::date_format2( $to );

            } break;
        }

        return false;
    }

    public function svcBillDescription($nItemID, $nSvcID, $aData = false, array &$aSvcSettings = array())
    {
        switch ($nSvcID)
        {
            case self::SERVICE_VIP: {
                if (empty($aData['link'])){
                    $aItemData = $this->model->itemData($nItemID, array('link'));
                    if ( ! empty($aItemData['link'])){
                        $aData['link'] = $aItemData['link'];
                    }
                }
                return _t('items', 'Приоритетное размещение объекта [a]#'.$nItemID.'[b]', array(
                        'a'=>'<a href="'.Items::url('view', $aData['link']).'" class="bill-items-item-link" data-item="'.$nItemID.'">',
                        'b'=>'</a>'
                    ));
            } break;
        }
        return '';
    }

    public function svcCron() # раз в час
    {
        if ( ! bff::cron() ) return;

        $sNow = $this->db->now();

        # Приоритетное размещение
        # Деактивируем услугу, с истекшим сроком действия
        $this->db->exec('UPDATE '.TABLE_ITEMS.'
            SET vip = 0
            WHERE vip = 1 AND vip_expire <= :now',
            array(':now'=>$sNow));
    }

    /**
     * Обработка ситуации c необходимостью ре-формирования URL
     */
    public function onLinksRebuild()
    {
        $this->model->itemsLinksRebuild();
    }

    /**
     * Формирование списка директорий/файлов требующих проверки на наличие прав записи
     * @return array
     */
    public function writableCheck()
    {
        $res = array_merge(parent::writableCheck(), array(
            bff::path('items', 'images') => 'dir', # изображения
            bff::path('tmp', 'images')   => 'dir', # tmp
        ));

        return $res;
    }
}