<?php

class Realty extends RealtyBase
{
    public function route()
    {
        $prefix = ( ! bff::subdomainsEnabled('realty') ? 'realty/' : '' );
        $res = bff::route(array(
            $prefix.'(.*)-([\d]+)\.html'           => 'realty/view/id=\\2',
            $prefix.'search/(.*)/(.*)(\/|)'        => 'realty/search/cat=\\1&type=\\2',
            $prefix.'search/(.*)(\/|)'             => 'realty/search/cat=\\1',
            $prefix.'(add|edit|search|fav|promote)(.*)' => 'realty/\\1',
            $prefix.'agents(.*)'                   => 'realty/agents',
        ), true);
        if ($res['event'] === false || ! method_exists($this, $res['event']))
            $res['event'] = 'index';

        return $this->$res['event']();
    }

    /**
     * Главная: блок "Недвижимость"
     */
    public function indexBlock()
    {
        # категории
        $aData['cats'] = $this->db->select_key('SELECT C.id, CL.title, C.keyword
                          FROM '.TABLE_REALTY_CATEGORIES.' C, '.TABLE_REALTY_CATEGORIES_LANG.' CL
                          WHERE C.enabled = 1 '.$this->db->langAnd(true, 'C', 'CL').'
                          ORDER BY C.num', 'id');

        # типы
        $aData['types'] = $this->typesGet(true, array('T.id','T.keyword','T.title_'.LNG.' as t'));
        if ($this->regionsFilterEnabled())
        {
            # при фильтрации по региону - считаем счетчики в зависимости от города
            $aData['ctypes'] = $this->db->select('SELECT CT.cat_id, CT.type_id, COUNT(I.id) as items
                    FROM '.TABLE_REALTY_CATEGORIES_TYPES.' CT
                        LEFT JOIN '.TABLE_REALTY_ITEMS.' I ON
                                I.cat_id = CT.cat_id
                            AND I.status = :status
                            AND I.type_id = CT.type_id
                            AND I.city_id = :city
                    GROUP BY cat_id, type_id', array(
                        ':status' => self::STATUS_PUBLICATED,
                        ':city'   => Geo::cityID(),
                    ));
        } else {
            $aData['ctypes'] = $this->db->select('SELECT cat_id, type_id, items
                FROM '.TABLE_REALTY_CATEGORIES_TYPES);
        }
        $aCatTypes = array();
        foreach ($aData['ctypes'] as $v) {
            $catID = $v['cat_id'];
            if ( ! isset($aCatTypes[$catID])) {
                $aCatTypes[$catID] = array();
            }
            $aCatTypes[$catID][$v['type_id']] = $v['items'];
        } $aData['ctypes'] = $aCatTypes;

        # последние объявления
        foreach ($aData['types'] as &$type) {
            $type['items'] = $this->db->select('SELECT I.id, I.title, I.title_alt,
                    I.link, I.imgfav, I.imgcnt, I.price, I.price_curr,
                    R.title_'.LNG.' as city_title
                FROM '.TABLE_REALTY_ITEMS.' I,
                     '.TABLE_REGIONS.' R
                WHERE I.type_id = :type
                  AND I.status = :status
                  AND I.city_id = R.id
                    '.( $this->regionsFilterEnabled() ? ' AND I.city_id = '.Geo::cityID() : '' ).'
                ORDER BY I.id DESC
                LIMIT 3', array(':type'=>$type['id'], ':status'=>self::STATUS_PUBLICATED));
            if ( ! empty($type['items'])) {
                foreach ($type['items'] as &$v) {
                    $v['img'] = RealtyImages::url($v['id'], $v['imgfav'], RealtyImages::szSmall);
                    $v['url'] = static::url('view', $v['link']);
                    unset($v['id'], $v['link'], $v['imgfav']);
                } unset($v);
            } else {
                $type['items'] = array();
            }
        } unset($type);

        # ссылки
        $aData['links'] = Items::model()->categoriesLinks(19, array(112, 114));
        foreach ($aData['links'] as &$v) {
            $v['url'] = Items::url('map', array('cat'=>$v['keyword'])); unset($v['keyword']);
        } unset($v);
        array_unshift($aData['links'], array(
            'title' => _t('realty', 'Агентства недвижимости'),
            'url'   => static::url('agents.list'),
        ));

        return $this->viewPHP($aData, 'index.block');
    }
    
    /**
     * Недвижимость: главная
     */
    public function index()
    {
        $aData = $this->search_filter(true);

        # vip блок
        $aData['vipBlock'] = $this->vipBlock();

        $aData['latest'] = $this->db->select('
            SELECT I.id, I.link, I.title, I.title_alt, I.imgfav, I.imgcnt, I.price, I.price_curr, R.title_'.LNG.' as city_title
            FROM '.TABLE_REALTY_ITEMS.' I,
                 '.TABLE_REGIONS.' R
            WHERE I.status = '.self::STATUS_PUBLICATED.' AND I.city_id = R.id
                '.( $this->regionsFilterEnabled() ? ' AND I.city_id = '.Geo::cityID() : '' ).'
            ORDER BY I.id DESC
            LIMIT 3');
        if ( ! empty($aData['latest'])) {
            foreach ($aData['latest'] as &$v) {
                $v['img'] = RealtyImages::url($v['id'], $v['imgfav'], RealtyImages::szSmall);
                $v['url'] = static::url('view', $v['link']);
                unset($v['id'], $v['link'], $v['imgfav']);
            } unset($v);
        } else {
            $aData['latest'] = array();
        }

        $aData['news'] = Publications::i()->indexCompaniesInCatNewsBlock(static::agentsCategory());

        # SEO: Главная страница
        $this->urlCorrection(static::url('index'));
        $this->seo()->canonicalUrl(static::url('index', array(), true));
        $this->setMeta('index', array(
            'region' => Geo::filter('title'),
        ), $aData);

        $this->initRightblock(__FUNCTION__, array('filter'=>&$aData['filter']));
        return $this->viewPHP($aData, 'index');
    }
    
    /**
     * Поиск объявлений
     */
    public function search()
    {
        $aData = $this->search_filter(false);

        $f = &$aData['f'];
        $nCatID = $f['c'];
        if ( ! $nCatID) {
            if (Request::isAJAX()) {
                $this->ajaxResponse(Errors::RELOAD_PAGE);
            }
            $this->errors->error404();
        }
        $sql = array();
        $sql[] = 'I.status = '.self::STATUS_PUBLICATED;
        $sql[] = 'I.price_curr = CURR.id';
        $sql[] = 'I.cat_id = '.$nCatID;
        $seoNoIndex = false;

        # тип: Продажа, Аренда
        if ($f['t']) {
            $sql[] = 'I.type_id = '.$f['t'];
        }

        # регион: город
        if ( ! Geo::cityIsOne() ) {
            if ($f['rc'] > 0) {
                $sql[] = 'I.city_id = '.$f['rc'];
            }
            if (Geo::filterUrl('id') != $f['rc']) {
                $seoNoIndex = true;
            }
        }

        $seoResetCounter = sizeof($sql); # всю фильтрацию ниже скрываем от индексации

        if($f['rd'] > 0) { # регион: район
            $sql[] = 'I.district_id = '.$f['rd'];
        }

        # Цена (в основной валюте)
        if ($f['pf'] > 0 || $f['pt'] > 0)
        {
            if ($f['pf'] > 0) { # от
                $sql[] = '(I.price * CURR.rate) >= '.$f['pf'];
                if ($f['pt'] > 0 && $f['pt'] > $f['pf']) {
                    $sql[] = '(I.price * CURR.rate) <= '.$f['pt']; # если до, и до больше от
                }
            } else if ($f['pt'] > 0) { # до
                $sql[] = '(I.price * CURR.rate) <= '.$f['pt'];
            }
        }
        if ($f['ph']) { # есть фотографии
            $sql[] = 'I.imgcnt > 0';
        }

        $sqlDP = $this->dp()->prepareSearchQuery($f['f'], false, $this->dpSettings($nCatID), 'I.');
        if( ! empty($sqlDP)) {
            $sql[] = $sqlDP;
        }
        
        $nPerpage = config::sys('realty.search.perpage', 10);;
        $nTotal = $this->db->one_data('SELECT COUNT(I.id) FROM '.TABLE_REALTY_ITEMS.' I, '.TABLE_CURRENCIES.' CURR WHERE '.join(' AND ', $sql));
        if ($f['cnt']) {
            $this->ajaxResponse(array('total'=>$nTotal));
        }

        $aCategory = $this->model->categoryDataSearch($nCatID);
        if (empty($aCategory)){
            $this->errors->error404();
        }
        $aType = array('id'=>'','title'=>'','keyword'=>'');
        if ($f['t']) {
            $aType = $this->model->typeDataSearch($f['t']);
            if (empty($aType)){
                $this->errors->error404();
            }
        }

        if(static::premoderation()){
            $sql[] = 'I.moderated > 0';
        }

        $aRequestURI = parse_url( Request::uri() );
        $sQuery = ( ! empty($aRequestURI['query']) ? trim( preg_replace('/&?page=(\d+)/', '', $aRequestURI['query']), '&').'&' : '' );
        $aData['pgn'] = $this->generatePagenationDots($nTotal, $nPerpage, 2, static::url('search', array('cat'=>$aCategory['keyword'], 'type'=>$aType['keyword']))."?{$sQuery}page={page}", $sqlLimit);
        $nPage = $this->input->get('page', TYPE_UINT); if (!$nPage) $nPage = 1;
        
        $nUserID = User::id();
        $aData['items'] = $this->db->select('SELECT I.id, I.link, I.title, I.descshort, I.imgcnt, I.imgfav,
                                    I.price, I.price_curr, I.price_params as pricep,
                                    I.svc, (I.svc & '.self::SERVICE_FIX.') as fixed, (I.svc & '.self::SERVICE_MARK.') as marked,
                                    '.($nUserID ? ' (F.item_id)' : '0').' as fav
                                FROM '.TABLE_REALTY_ITEMS.' I
                                       '.($nUserID ? ' LEFT JOIN '.TABLE_REALTY_FAV.' F ON I.id = F.item_id AND F.user_id = '.$nUserID : '').',
                                     '.TABLE_CURRENCIES.' CURR
                                WHERE '.join(' AND ', $sql).'
                                ORDER BY fixed DESC, I.fixed_order DESC, I.publicated_order DESC
                                '.$sqlLimit);

        # vip блок
        $aData['vipBlock'] = $this->vipBlock($nCatID, $f['t'], $f['rc']);

        # SEO: Поиск объявлений
        $this->seo()->robotsIndex(!(sizeof($sql) - $seoResetCounter) && !$seoNoIndex);
        $this->urlCorrection(static::url('search', array('cat'=>$aCategory['keyword'], 'type'=>$aType['keyword'])));
        $this->seo()->canonicalUrl(static::url('search', array('cat'=>$aCategory['keyword'], 'type'=>$aType['keyword']), true), array(
            'page' => $nPage
        ));
        $this->setMeta('search', array(
            'category' => $aCategory['title'],
            'type'     => mb_strtolower($aType['title']),
            'region'   => Geo::filter('title'),
            'page'     => $nPage,
        ), $aCategory);
        $aData['cat'] = &$aCategory;

        $this->initRightblock(__FUNCTION__, array('filter'=>&$aData['filter']));
        return $this->viewPHP($aData, 'search');
    }
    
    protected function search_filter($isIndex = false)
    {
        $f = $this->input->postgetm(array(
            'rc' => TYPE_INT,   # регион: город
            'rd' => TYPE_UINT,  # регион: район
            'c'  => TYPE_UINT,  # категория: Квартира, Комната
            't'  => TYPE_UINT,  # тип: Продажа, Аренда
            'f'  => TYPE_ARRAY, # дин. свойства
            'pf' => TYPE_PRICE, # цена, от
            'pt' => TYPE_PRICE, # цена, до
            'ph' => TYPE_BOOL,  # есть фотографии
            'cnt' => TYPE_BOOL, # только подсчет кол-ва
        ));

        if ($f['cnt']) {
            return array('f' => $f);
        }

        if ($f['rc'] === 0) {
            $f['rc'] = ( $this->regionsFilterEnabled() ? Geo::cityID() : -1 );
        }

        if ( ! $f['c'] && ! $isIndex) {
            $f['c'] = $this->model->categoryIDByKeyword($this->input->get('cat', TYPE_NOTAGS));
        }
        $nCatID = $f['c'];

        if ( ! $f['t'] && ! $isIndex) {
            $f['t'] = $this->model->typeIDByKeyword($this->input->get('type', TYPE_NOTAGS));
        }

        # район города
        $aData['districts'] = Geo::districtOptions($f['rc'], $f['rd'], ($f['rc'] ? _t('realty','Все районы') : _t('','Выберите город')) );

        # категории
        $aData['cats'] = $this->db->select('SELECT C.id, C.keyword, CL.title
                          FROM '.TABLE_REALTY_CATEGORIES.' C, '.TABLE_REALTY_CATEGORIES_LANG.' CL
                          WHERE C.enabled = 1 '.$this->db->langAnd(true, 'C', 'CL').'
                          ORDER BY C.num');
        $aData['cats'] = func::array_transparent($aData['cats'], 'id', true);
        
        $aData['cats_options'] = '';
        foreach ($aData['cats'] as $v) {
            $aData['cats_options'] .= '<option'.($nCatID == $v['id'] ? ' selected="selected"' : '').' value="'.$v['id'].'">'.$v['title'].'</option>';
        }
        # типы
        $aData['types']  = $this->typesGet( true, array('T.id','T.keyword','T.title_'.LNG.' as t') );
        if ($this->regionsFilterEnabled() && $isIndex)
        {
            # при фильтрации по региону - считаем счетчики в зависимости от города
            $aData['ctypes'] = $this->db->select('SELECT CT.cat_id, CT.type_id, COUNT(I.id) as items
                    FROM '.TABLE_REALTY_CATEGORIES_TYPES.' CT
                        LEFT JOIN '.TABLE_REALTY_ITEMS.' I ON
                                I.cat_id = CT.cat_id
                            AND I.status = :status
                            AND I.type_id = CT.type_id
                            AND I.city_id = :city
                    GROUP BY cat_id, type_id', array(
                        ':status' => self::STATUS_PUBLICATED,
                        ':city'   => Geo::cityID(),
                    ));
        } else {
            $aData['ctypes'] = $this->db->select('SELECT cat_id, type_id, items
                FROM '.TABLE_REALTY_CATEGORIES_TYPES);
        }
        $aCatTypes = array();
        foreach ($aData['ctypes'] as $v) {
            $catID = $v['cat_id'];
            if ( ! isset($aCatTypes[$catID])) {
                $aCatTypes[$catID] = array();
            }
            $aCatTypes[$catID][$v['type_id']] = $v['items'];
        } $aData['ctypes'] = $aCatTypes;
        
        # дин. свойства
        $aData['dp'] = $this->dpForm($nCatID, true, array(), array('f'=>$f['f']));
        
        # активируем меню
        bff::setActiveMenu('//realty/search', ! $isIndex, false);

        $aData['f'] = &$f;
        return array('filter' => $this->viewPHP($aData, 'search.filter'),
                     'f'      => $f,
                     'cats'   => $aData['cats'],
                     'types'  => $aData['types'],
                     'ctypes' => $aData['ctypes']);
    }
    
    /**
     * Просмотр объявления
     */
    public function view()
    {
        if (Request::isAJAX())
        {
            $aResponse = array();
            
            switch ($this->input->post('act', TYPE_STR)) {
                case 'photo_view': { # подробный просмотр фотографий ОБ
                    $aRes = $this->initImages()->processImagesView('');
                    $aResponse = array_merge($aRes, $aResponse);
                } break;
            }
            $aResponse['res'] = $this->errors->no();
            $this->ajaxResponse($aResponse);
        }
                                                                                                       
        $nItemID = $this->input->get('id', TYPE_UINT);
        $nUserID = User::id();
        $isPrint = $this->input->get('print', TYPE_BOOL);

        bff::setActiveMenu('//realty/search', true);

        if ( ! $nItemID) {
            $this->errors->error404();
        }
        $aData = $this->db->one_array('SELECT I.id, I.link, I.address_full, I.descfull, I.imgcnt, I.imgfav, I.img,
                                I.blocked_reason, I.title_alt, I.company_id,
                                I.info, I.status, '.($nUserID ? ' (F.item_id)' : '0').' as fav,
                                I.user_id, I.user_type, I.user_name, I.user_email, I.user_phone,
                                I.price, I.price_curr, I.price_params, I.lat, I.lng, I.views,
                            C.address as cat_address, I.type_id, CT.prices as type_prices, CT.view_title_'.LNG.' AS view_title
                        FROM '.TABLE_REALTY_ITEMS.' I
                               '.($nUserID>0 ? ' LEFT JOIN '.TABLE_REALTY_FAV.' F ON F.item_id = I.id
                                AND F.user_id = '.$nUserID : '').',
                             '.TABLE_REALTY_CATEGORIES.' C,
                             '.TABLE_REALTY_CATEGORIES_TYPES.' CT
                        WHERE I.id = :item
                          AND C.id = I.cat_id
                          AND CT.cat_id = I.cat_id
                          AND CT.type_id = I.type_id
                        ', array(':item'=>$nItemID));
        if (empty($aData) || $aData['status'] == self::STATUS_NEW) {
            $this->errors->error404();
        }
        
        if ($aData['status'] == self::STATUS_BLOCKED) {
            return $this->showForbidden(_t('','Объявление №[id]', array('id'=>$nItemID)),
                _t('', 'Заблокировано по причине:<br/><b>[reason]</b>', array('reason'=>nl2br($aData['blocked_reason']))));
        }
        else if ($aData['status'] == self::STATUS_PUBLICATED_OUT) {
            return $this->showForbidden(_t('','Объявление №[id]', array('id'=>$nItemID)),
                _t('', 'Срок публикации объявления истек'));
        }

        # Изображения
        $aData['img'] = $this->initImages()->prepareView($nItemID, $aData['imgcnt'], $aData['img']);

        # SEO: Просмотр объявления
        $aData['url'] = static::url('view', $aData['link']);
        $this->urlCorrection($aData['url']);
        $this->seo()->robotsIndex(!$isPrint);
        $this->seo()->canonicalUrl($aData['link']);
        $this->setMeta('view', array(
            'title'       => $aData['title_alt'],
            'description'     => strip_tags($aData['address_full']),
            //'description' => tpl::truncate(strtr(strip_tags($aData['descfull']), array("\r\n"=>', ',"\n"=>', ')), 150),
            'region'      => Geo::filter('title'),
        ), $aData);
        if ( ! $isPrint) {
            $seoShareImages = array();
            foreach ($aData['img'] as $v) {
                $seoShareImages[] = $v['view'];
            }
            $this->seo()->setSocialMetaOG($aData['share_title'], $aData['share_description'], $seoShareImages, $aData['url'], $aData['share_sitename']);
        }

        if ($isPrint) {
            View::setLayout('print');
            return $this->viewPHP($aData, 'view.print');
        }

        if ( ! $nUserID || ($aData['user_id'] > 0 && $aData['user_id']!=$nUserID)) {
            $this->db->exec('UPDATE '.TABLE_REALTY_ITEMS.' SET views = views + 1 WHERE id = '.$nItemID);
        }
        if ($aData['company_id'] > 0) {
            $aData['company_data'] = Items::model()->companyData($aData['company_id']);
        }

        $aData['others'] = $this->db->select('SELECT I.id, I.imgcnt, I.imgfav, I.title_alt, I.link, I.descshort, I.address_full, I.price, I.price_curr, I.price_params
                            FROM '.TABLE_REALTY_ITEMS.' I
                            WHERE I.status = '.self::STATUS_PUBLICATED.' AND I.id != :item
                                '.( $this->regionsFilterEnabled() ? ' AND I.city_id = '.Geo::cityID() : '' ).'
                            ORDER BY I.publicated_order DESC
                            LIMIT 4', array(':item'=>$nItemID));
        if ( ! empty($aData['others'])) {
            foreach ($aData['others'] as &$v) {
                $v['imgfav'] = $this->initImages()->getUrl($v['id'], $v['imgfav'], RealtyImages::szSmall);
                $v['link'] = static::url('view', $v['link']);
            } unset($v);
        }

        $aData['from_history'] = (strpos(Request::referer(), static::url('search')) !== false ? 1 : 0);
        $this->initRightblock(__FUNCTION__);
        return $this->viewPHP($aData, 'view');
    }
    
    /**
    * Добавление объявления
    */
    public function add()
    {
        if ( ! empty($_GET['finish'])) {
            $aData['id'] = $this->input->get('id', TYPE_UINT);
            $aItemData = $this->model->itemData($aData['id'], 'link');
            $aData['url_view'] = static::url('view', $aItemData['link']);
            bff::setMeta(_t('realty', 'Добавление объявления'));
            bff::setActiveMenu('//realty', true);
            $this->initRightblock(__FUNCTION__);
            return $this->viewPHP($aData, 'add.finish');
        }
        
        $aData = $this->input->postm(array(
            'cat_id'  => TYPE_UINT, // категория: Квартира, Комната...
            'type_id' => TYPE_UINT, // тип: Продажа, Аренда...
            'price'        => TYPE_PRICE, // цена
            'price_params' => TYPE_ARRAY_UINT, // цена: параметры; торг...
            'price_curr'   => TYPE_UINT, // цена: валюта
            'city_id'      => TYPE_UINT, // город
            'district_id'  => TYPE_UINT, // район
            'address'      => TYPE_NOTAGS, // адрес
            'filename'     => TYPE_ARRAY_NOTAGS, // фотографии
            'info'         => array(TYPE_NOTAGS, 'len'=>10000), // доп. информация
            'user_type'    => TYPE_UINT,   // user: тип
            'user_name'    => array(TYPE_NOTAGS, 'len'=>100), // user: имя
            'user_phone'   => array(TYPE_NOTAGS, 'len'=>100), // user: телефон
            'user_email'   => array(TYPE_NOTAGS, 'len'=>100), // user: email
            'member'       => TYPE_BOOL,   // уже зарегистрирован
            'pass'         => TYPE_NOTRIM, // пароль для авторизации: user_email/pass
            'captcha'      => TYPE_STR,    // капча
            'period'       => TYPE_UINT,   // период публикации
            'lat'          => TYPE_NUM, // адрес, координата LAT
            'lng'          => TYPE_NUM, // адрес, координата LNG
            'user_type_c'  => TYPE_UINT,   //частное лицо/организация
            'company_id'   => TYPE_UINT, //Объявление компании
        ));
        if (Request::isAJAX())
        {
            $aCategories = $this->categoriesGet( true, array('C.id','CL.title','C.address','CL.tpl_descshort','CL.tpl_descfull') );

            $aResponse = array('status'=>0, 'user'=>false, 'captcha'=>false);
            
            do 
            {
                $sEmail = $aData['user_email'];
                
                $nUserStatus = false;

                if ( ! User::id())
                {
                    # для неавторизованного пользователя - email обязателен
                    if( ! $this->input->isEmail($sEmail, false)) {
                        $this->errors->set( _t('', 'E-mail адрес указан некорректно'), 'user_email' );
                    }
                    if($aData['member']) {
                        # авторизуем
                        if (empty($aData['pass'])) {
                            $this->errors->set( _t('users', 'Укажите пароль'), 'pass' );
                        } else {
                            $res = $this->users()->userAuth($sEmail, 'email', $aData['pass'], false, true);
                            if($res === true) {
                                # успешно авторизовали
                                $aResponse['user'] = User::id();
                            } elseif($res === 1) {
                                # пользователь неактивирован
                                $aUserData = config::get('__user_preactivate_data', array());
                                $aResponse['user'] = $aUserData['id'];
                                $nUserStatus = 'na';
                            }
                        }
                    } else {
                        $aUser = Users::model()->userDataByFilter(array('email'=>$sEmail), array('user_id', 'blocked', 'activated'));
                        if( ! empty($aUser)) {
                            $this->errors->set( _t('users', 'Пользователь с указанным email адресом уже зарегистрирован') );
                            $aResponse['user'] = -1; # пользователь с указанным email адресом уже существует
                        } else {
                            # проверяем капчу
                            if (empty($aData['captcha']) || ! CCaptchaProtection::correct($this->input->cookie('c2'), $aData['captcha']) ) {
                                $this->errors->set( _t('', 'Результат с картинки указан некорректно'), 'captcha' );
                                $aResponse['captcha'] = true;
                            } else {
                                $aResponse['user'] = 1;                                                        
                                $nUserStatus = 'new'; 
                            }
                        }
                    }
                               
                    if( ! $this->errors->no()) {
                        break;
                    }
                } else {
                    $aResponse['user'] = User::id();
                    $nUserStatus = 'ok';
                    if( ! $this->security->validateToken()) {
                        $this->errors->reloadPage();
                        break;
                    }
                }

                $nCatID = $aData['cat_id'];
                if( ! $nCatID || empty($aCategories[$nCatID])) {
                    $this->errors->set( _t('realty', 'Укажите тип недвижимости') );
                }
                if( ! $aData['type_id']) {
                    $this->errors->set( _t('realty', 'Укажите тип сделки') );
                }
                if( ! $aData['price']) {
                    $this->errors->set( _t('realty', 'Укажите цену'), 'price' );
                }
                if( ! $aData['price_curr']) {
                    $this->errors->set( _t('realty', 'Укажите валюту') );
                }
                
                if ($nCatID && $aCategories[$nCatID]['address'] > 0) {
                    if( ! Geo::cityIsValid($aData['city_id']) ) {
                        $this->errors->set( _t('realty', 'Укажите город') );
                    }
                }
                
                if( ! in_array($aData['user_type'], array(self::USERTYPE_AGENT, self::USERTYPE_OWNER))) {
                    $this->errors->set( _t('realty', 'Укажите кто вы, Владелец или Агент') );
                }
                
                if (empty($aData['user_name'])) {
                    $this->errors->set( _t('realty', 'Укажите ваше имя'), 'user_name' );
                }
                if (empty($aData['user_phone'])) {
                    $this->errors->set( _t('realty', 'Укажите ваше телефон'), 'user_phone' );
                }
            
                if( ! $this->errors->no()) break;

                $aData['status'] = self::STATUS_PUBLICATED;
                $aData['status_prev'] = self::STATUS_NEW;

                if ($nUserStatus == 'new')
                {
                    # регистрируем нового пользователя
                    $aUserData = $this->users()->userRegister(array(
                        'email' => $sEmail,
                        'name'  => $aData['user_name'],
                        'phone' => $aData['user_phone'],
                    ));
                    if (!empty($aUserData['user_id']))
                    {
                        $nUserID = $aUserData['user_id'];
                        # добавляем возможность получить повторное email-уведомление
                        $this->users()->setRegisterSecondEnotifyFlag($nUserID);
                        # отправляем email уведомление
                        bff::sendMailTemplate(array(
                            'name'     => $aData['user_name'],
                            'password' => $aUserData['password'],
                            'email'    => $sEmail,
                            'activate_link' => $aUserData['activate_link']
                        ), 'member_registration', $sEmail);

                        # переносим публикацию объявления на момент активации пользователя
                        $aData['status'] = self::STATUS_NEW;
                    } else {
                        $nUserID = 0;
                        $this->errors->set( _t('realty', 'Ошибка регистрации') );
                        break;
                    }                    
                } else {
                    $nUserID = User::id();
                }
                $aResponse['user'] = $nUserStatus;

                $sPublicateTo = $this->preparePublicatePeriodTo($aData['period'], time());

                # проверим company_id
                if (User::id() && $aData['user_type_c'] == 2 && $aData['company_id'] > 0) {
                    $aComp = Users::model()->companyInfo(User::id(), $aData['company_id']);
                    if (empty($aComp)) {
                        $aData['company_id'] = 0;
                    } else {
                        $aData['user_type'] = self::USERTYPE_AGENT;
                    }
                } else {
                    $aData['company_id'] = 0;
                }

                # сохраняем объявление + публикуем (если пользователь авторизован)
                $nItemID = $this->model->itemSave(0, array(
                    'user_id' => $nUserID,
                    'cat_id'  => $nCatID,
                    'type_id' => $aData['type_id'],
                    'city_id' => $aData['city_id'],
                    'district_id' => $aData['district_id'],
                    'lat' => $aData['lat'],
                    'lng' => $aData['lng'],
                    'address' => $aData['address'],
                    'info' => $aData['info'],
                    'price' => $aData['price'],
                    'price_params' => $aData['price_params'],
                    'price_curr'  => $aData['price_curr'],
                    'filename'    => $aData['filename'],
                    'user_name'   => $aData['user_name'],
                    'user_email'  => $sEmail,
                    'user_phone'  => $aData['user_phone'],
                    'user_type'   => $aData['user_type'],
                    'status'      => $aData['status'],
                    'status_prev' => $aData['status_prev'],
                    'publicated'  => $this->db->now(),
                    'publicated_order' => $this->db->now(),
                    'publicated_to' => $sPublicateTo,
                    'moderated' => 0,
                    'company_id' => $aData['company_id'],
                ), 'd');

                if (!$nItemID) {
                    $this->errors->set( _t('realty', 'Не удалось опубликовать объявление, возникла системная ошибка, сообщите администрации.') );
                    break;
                } else {
                    $this->initImages()->renameImageFileCustom($nItemID, $aData['filename']);
                    $this->security->userCounter('realty', 1, (User::id() > 0 ? false : $nUserID) );
                    $aResponse['status']  = 2; # объявление обновлено в базе
                    if ( $aData['status'] == self::STATUS_PUBLICATED && ! static::premoderation()) {
                        $this->itemsCounterUpdate($aData['cat_id'], $aData['type_id'], true);
                    }
                }
                $aResponse['id'] = $nItemID;
                
                if($nUserStatus == 'new') {
                    Request::deleteCOOKIE('c2');
                }

            } while(false);

            $this->ajaxResponseForm($aResponse);
        }
        
        $aData['cats'] = $this->categoriesGet(true, array('C.id','CL.title AS t','C.address AS ad'));
        $aData['types']  = $this->typesGet( true, array('T.id','T.title_'.LNG.' AS t') );
        $aData['ctypes'] = $this->db->select('SELECT cat_id, type_id, prices FROM '.TABLE_REALTY_CATEGORIES_TYPES);
        foreach($aData['ctypes'] as $v) {
            if(isset($aData['cats'][$v['cat_id']])) {
                $aData['cats'][$v['cat_id']]['tp'][$v['type_id']] = $v['prices'];
            }
        } unset($aData['ctypes']);
        
        $aData['curr']  = Site::currencyOptions($aData['price_curr'], false);
        $aData['hash'] = $this->security->getToken();
        if (User::id()) {
            $aData['agentCompanies'] = Users::model()->agentCompanies(User::id());
        } else {
            $aData['agentCompanies'] = array();
        }
        $aData['user_type_c'] = 1;

        # SEO: Добавление объявления
        $this->urlCorrection(static::url('add'));
        $this->seo()->canonicalUrl(static::url('add', array(), true));
        $this->setMeta('add', array('region' => Geo::filter('title')), $aData);

        bff::showInstruction('realty_add');
        bff::setActiveMenu('//realty');
        $this->initRightblock(__FUNCTION__);
        return $this->viewPHP($aData, 'add');
    }
    
    /**
     * Редактирование объявления
     */
    public function edit()
    {
        $isAjax = Request::isAJAX();

        bff::setMeta(_t('realty', 'Редактирование объявления'));
        bff::setActiveMenu('//realty', true);
        
        $nItemID = $this->input->getpost('id', TYPE_UINT);
        if( ! $nItemID) {
            if($isAjax) $this->ajaxResponse(Errors::RELOAD_PAGE);
            $this->errors->error404();
        }
        
        $aData = $this->db->one_array('SELECT I.*, CT.prices as cat_prices, T.title_'.LNG.' as type_title,
                                R.title_'.LNG.' as city_title, R.ycoords as city_ycoords
                              FROM '.TABLE_REALTY_ITEMS.' I,
                                   '.TABLE_REALTY_TYPES.' T,
                                   '.TABLE_REGIONS.' R,
                                   '.TABLE_REALTY_CATEGORIES_TYPES.' CT
                              WHERE I.id = :item
                                AND I.type_id = T.id 
                                AND I.city_id = R.id
                                AND I.cat_id  = CT.cat_id AND I.type_id = CT.type_id', array(':item'=>$nItemID));
 
        if (empty($aData) || $aData['status'] == self::STATUS_NEW) {
            if($isAjax) $this->ajaxResponse(Errors::RELOAD_PAGE);
            $this->errors->error404();
        }

        if ( ! User::isCurrent($aData['user_id'])) {
            if($isAjax) $this->ajaxResponse(Errors::RELOAD_PAGE);
            return $this->showForbidden( _t('realty', 'Редактирование объявления'), _t('', 'Вы не явлетесь владельцем данного объявления'));
        }

        if ($aData['status'] == self::STATUS_BLOCKED && ! $aData['moderated']) {
            if(Request::isAJAX()) $this->ajaxResponse(Errors::RELOAD_PAGE);
            if( ! empty($_GET['success'])) {
                $this->redirect( Users::url('my.profile.tab', array('tab'=>'adv', 'subtab'=>'realty')) );
            }
            return $this->showForbidden( _t('', 'Объявление заблокировано'), _t('', 'Причина:<br/><b>[reason]</b>
                <br/> <span class="red">Объявление ожидает проверки модератора</span>', array('reason'=>nl2br($aData['blocked_reason']))));
        }

        $aCat = $this->db->one_array('SELECT C.id, C.address, CL.title
            FROM '.TABLE_REALTY_CATEGORIES.' C,
                 '.TABLE_REALTY_CATEGORIES_LANG.' CL
            WHERE C.id = :id '.$this->db->langAnd(true, 'C', 'CL'), array(':id'=>$aData['cat_id']));
        $aData['ch'] = crc32($aData['img'] . $aData['info']);
        
        if (Request::isAJAX())
        {
            $aResponse = array();
            
            $this->input->postm(array(
                'price'        => TYPE_PRICE, // цена
                'price_params' => TYPE_ARRAY_UINT, // цена: параметры; торг...
                'price_curr'   => TYPE_UINT, // цена: валюта
                'city_id'      => TYPE_UINT, // город
                'district_id'  => TYPE_UINT, // район
                'address'      => TYPE_NOTAGS, // адрес
                'filename'     => TYPE_ARRAY_NOTAGS, // фотографии
                'info'         => array(TYPE_NOTAGS, 'len'=>10000), // доп. информация
                'user_type'    => TYPE_UINT,   // user: тип
                'user_name'    => array(TYPE_NOTAGS, 'len'=>100), // user: имя
                'user_phone'   => array(TYPE_NOTAGS, 'len'=>100), // user: телефон
                'user_email'   => array(TYPE_NOTAGS, 'len'=>100), // user: email
                'ch'           => TYPE_STR,  // crc данных, для проверки изменений
                'lat'          => TYPE_NUM,  // адрес, координата LAT
                'lng'          => TYPE_NUM,  // адрес, координата LNG
                'user_type_c'  => TYPE_UINT, // частное лицо/организация
                'company_id'   => TYPE_UINT, // объявление компании

            ), $p); extract($p, EXTR_REFS);
            
            do {
                
                if( ! $price) {
                    $this->errors->set( _t('realty', 'Укажите цену'), 'price' );
                }
                if( ! $price_curr) {
                    $this->errors->set( _t('realty', 'Укажите валюту') );
                }
                if($aCat['address'] > 0 && ! Geo::cityIsValid($city_id)) {
                    $this->errors->set( _t('realty', 'Укажите город') );
                }
                if( ! in_array($user_type, array(self::USERTYPE_AGENT, self::USERTYPE_OWNER))) {
                    $this->errors->set( _t('realty', 'Укажите кто вы, Владелец или Агент') );
                }
                if (empty($user_name)) {
                    $this->errors->set( _t('realty', 'Укажите ваше имя'), 'user_name' );
                }
                if (empty($user_phone)) {
                    $this->errors->set( _t('realty', 'Укажите ваше телефон'), 'user_phone' );
                }
                
                if( ! $this->security->validateToken()) {
                    $this->errors->reloadPage();
                    break;
                }

                if( ! $this->errors->no()) break;

                # проверим company_id
                if (User::id() && $user_type_c == 2 && $company_id > 0) {
                    $aComp = Users::model()->companyInfo(User::id(), $company_id);
                    if (empty($aComp)) {
                        $company_id = 0;
                    } else {
                        $user_type = self::USERTYPE_AGENT;
                    }
                } else {
                    $company_id = 0;
                }

                $sqlUpdate = array(
                    'cat_id'     => $aData['cat_id'], // для сохранения дин. свойств
                    'type_id'    => $aData['type_id'], // для сохранения дин. свойств
                    'city_id'    => $city_id,
                    'district_id' => $district_id,
                    'address'    => $address,
                    'info'       => $info,
                    'price'      => $price,
                    'price_params' => $price_params,
                    'price_curr' => $price_curr,
                    'filename'   => $filename,
                    'user_name'  => $user_name,
                    'user_email' => $user_email,
                    'user_phone' => $user_phone,
                    'user_type'  => $user_type,
                    'lat'        => $lat,
                    'lng'        => $lng,
                    'company_id' => $company_id,
                );

                # отправляем на повторную модерацию
                # если изменялись фотографии или описание
                if( crc32(join(',',$filename) . $info) != $aData['ch'] ) {
                    if ($aData['moderated']) {
                        $sqlUpdate['moderated'] = 2;
                    }
                }

                # обновляем объявление
                $this->model->itemSave($nItemID, $sqlUpdate, 'd');
            
            } while(false);

            $this->ajaxResponseForm($aResponse);
        }

        $aData['cat']  = &$aCat;
        $aData['curr'] = Site::currencyOptions($aData['price_curr'], false);

        if ($aData['city_id']) {
            $aData['districts'] = Geo::districtList($aData['city_id']);
        }
        $aData['dp'] = $this->dpForm($aCat['id'], false, $aData);
        $aData = HTML::escape($aData, 'html', array('user_name', 'user_phone', 'user_email'));

        $aData['hash'] = $this->security->getToken();
        if (User::id()){
            $aData['agentCompanies'] = Users::model()->agentCompanies(User::id());
        } else {
            $aData['agentCompanies'] = array();
        }
        $aData['user_type_c'] = $aData['company_id'] > 0 ? 2 : 1;

        bff::showInstruction('realty_edit');
        $this->initRightblock(__FUNCTION__);
        return $this->viewPHP($aData, 'edit');
    }
    
    /**
     * Список избранных объявлений
     */
    public function fav()
    {
        bff::setActiveMenu('//realty/fav');
        bff::setMeta(_t('','Избранные объявления'));
        if ( ! ($userID = User::id())) {
            return $this->showForbidden(_t('','Избранные объявления'), _t('','Возможность добавлять объявления в избранные доступна только после авторизации.'), true);
        }
        
        $aData['items'] = $this->db->select('SELECT I.id, I.link, I.imgfav, I.address, I.descshort, I.price, I.price_curr, I.price_params as pricep
                            FROM '.TABLE_REALTY_ITEMS.' I, '.TABLE_REALTY_FAV.' F
                            WHERE F.user_id = '.$userID.' AND F.item_id = I.id
                                AND I.status = '.self::STATUS_PUBLICATED.'
                            ');

        $this->security->userCounterCheck('realty_fav', sizeof($aData['items']));
        $this->initRightblock(__FUNCTION__);
        return $this->viewPHP($aData, 'fav');
    }
    
    /**
     * Список объявлений пользователя
     */
    public function profile()
    {
        $userID = User::id();
        if( ! $userID)
            return '';

        $nPerpage = config::sys('realty.profile.perpage', 10);
        $nTotal = $this->model->itemsListProfile($userID, true);
        $aData['pgn'] = $this->generatePagenationDots($nTotal, $nPerpage, 3, Users::url('my.profile.tab', array('tab'=>'adv','subtab'=>'realty')).'&page={page}', $sqlLimit);

        $aData['items'] = $this->model->itemsListProfile($userID, false, $sqlLimit);
        if ( ! empty($aData['items'])) {
            foreach ($aData['items'] as &$v) {
                $v['url'] = static::url('view', $v['link']);
                $v['img'] = RealtyImages::url($v['id'], $v['imgfav'], RealtyImages::szList);
            } unset($v);
        }

        $this->security->userCounterCheck('realty', $nTotal);
        bff::setMeta(_t('', 'Мои объявления - Кабинет'));
        $this->initRightblock(__FUNCTION__);
        return $this->viewPHP($aData, 'profile');
    }

    /**
     * Список объявлений компании (Недвижимость)
     * @param integer $companyID ID компании
     * @param array $companyData @ref данные о компании
     */
    public function company_items($companyID = 0, array &$companyData)
    {
        $perpage = config::sys('realty.company.perpage', 10);
        $total = $this->model->itemsListCompany($companyID, true);
        $sqlLimit = '';
        $data['pgn'] = $this->generatePagenationDots($total, $perpage, 3,
            Items::url('view', array('link'=>$companyData['link'], 'tab'=>'board', 'q'=>$companyData['st'].'&page={page}')),
            $sqlLimit);

        $data['items'] = $this->model->itemsListCompany($companyID, false, $sqlLimit);
        if (!empty($data['items'])) {
            foreach ($data['items'] as &$v) {
                $v['url'] = static::url('view', $v['link']);
                $v['img'] = RealtyImages::url($v['id'], $v['imgfav'], RealtyImages::szList);
            } unset($v);
        }

        return $this->viewPHP($data, 'company');
    }

    /**
     * Странице успешной активации услуги
     */
    public function promote()
    {
        if( ! bff::servicesEnabled()) return false;

        $nItemID = $this->input->get('id', TYPE_UINT);
        $nSvcID = $this->input->get('svc', TYPE_UINT);

        $aSvc = $this->svc()->model->svcData($nSvcID);
        if ($nItemID && ! empty($aSvc) && $aSvc['module'] == $this->module_name)
        {
            $this->seo()->robotsIndex(false);
            bff::setMeta(_t('svc', 'Активация услуги'));
            bff::setActiveMenu('//realty/search', true);
            $aItem = $this->model->itemData($nItemID, array('id','title','link'));
            if (empty($aItem)) {
                $this->errors->error404();
            }
            $this->initRightblock(__FUNCTION__);
            return $this->showSuccess(
                _t('svc', 'Активация услуги'),
                _t('realty', 'Для вашего объявления "[item_link]"<br />была успешно активирована услуга "[svc_title]"', array(
                    'item_link'=>'<a href="'.static::url('view', $aItem['link']).'">'.$aItem['title'].'</a>',
                    'svc_title'=>$aSvc['title']))
            );
        } else {
            $this->redirect( static::url('index') );
        }
    }

    /**
     * Список "Агентства недвижимости"
     */
    public function agents()
    {
        $categoryID = static::agentsCategory();
        $sql = 'I.id = C.item_id AND C.category_id = '.$categoryID.'
            AND I.enabled = 1 AND I.moderated = 1 AND I.id = IA.item_id';
        if( $this->regionsFilterEnabled() ) {
            $sql .= ' AND IA.city_id = '.Geo::cityID();
        }

        $nPerpage = config::sys('realty.agents.perpage', 10);
        $nTotal = $this->db->one_data('SELECT COUNT(I.id) FROM '.TABLE_ITEMS.' I, '.TABLE_ITEMS_ADDR.' IA, '.TABLE_ITEMS_IN_CATEGORIES.' C WHERE '.$sql);
        $aData['pgn'] = $this->generatePagenationDots($nTotal, $nPerpage, 2, static::url('agents.list').'?page={page}', $sqlLimit);
        $nPage = $this->input->get('page', TYPE_UINT); if (!$nPage) $nPage = 1;

        $aData['items'] = $this->db->select('
                SELECT I.id, I.link, IL.title, IL.descshort as description, IA.address, IA.phonesq as phones,
                       IA.site, I.img_list as img, I.imgcnt
                FROM '.TABLE_ITEMS.' I,
                     '.TABLE_ITEMS_LANG.' IL,
                     '.TABLE_ITEMS_ADDR.' IA,
                     '.TABLE_ITEMS_IN_CATEGORIES.' C
                WHERE '.$sql.$this->db->langAnd(true, 'I', 'IL').'
                GROUP BY I.id 
                ORDER BY I.vip DESC, C.num_order
                '.$sqlLimit);

        if( ! empty($aData['items']))
        {
            foreach($aData['items'] as &$v)
            {
                if( ! empty($v['phones'])) {
                    $v['phones'] = join('<br/>', explode(', ', $v['phones']));
                }
                $v['link'] = Items::url('view', $v['link']);
            } unset($v);
        }

        # SEO: Агентства
        $this->urlCorrection(static::url('agents.list'));
        $this->seo()->canonicalUrl(static::url('agents.list', array(), true), array(
            'page' => $nPage
        ));
        $this->setMeta('agents', array(
            'region' => Geo::filter('title'),
            'page'   => $nPage
        ), $aData);

        bff::setActiveMenu('//realty/agents');
        $this->initRightblock(__FUNCTION__);
        return $this->viewPHP($aData, 'agents');
    }

    public function ajax()
    {
        $aResponse = array();
        $nUserID = User::id();

        switch ($this->input->get('act'))
        {
            case 'fav': 
            {
                $nItemID = $this->input->post('id', TYPE_UINT);
                if( ! $nItemID || ! $nUserID) {
                    $this->ajaxResponse(Errors::IMPOSSIBLE);
                }
                
                if( ! $this->security->validateToken()) {
                    $this->ajaxResponse(Errors::IMPOSSIBLE);
                }
                
                $bAdded = false;
                $bExists = $this->db->one_data('SELECT item_id FROM '.TABLE_REALTY_FAV.'
                    WHERE user_id = '.$nUserID.' AND item_id = '.$nItemID.' LIMIT 1');
                if (empty($bExists)) {
                    $res = $this->db->insert(TABLE_REALTY_FAV, array('user_id'=>$nUserID, 'item_id'=>$nItemID), false);
                    if (empty($res)) {
                        $this->ajaxResponse(Errors::IMPOSSIBLE);
                    }
                    $bAdded = true;
                } else {
                    $this->db->delete(TABLE_REALTY_FAV, array('user_id'=>$nUserID, 'item_id'=>$nItemID));
                }
                
                $nCurrent = $this->security->userCounter('realty_fav', ($bAdded ? 1 : -1) );
                
                $this->ajaxResponse(array('res'=>$this->errors->no(), 'added'=>$bAdded, 'n'=>$nCurrent));
                                                          
            } break;
            case 'dp-form': 
            {
                $nCatID = $this->input->postget('cat_id', TYPE_UINT);
                if( ! $nCatID) $this->ajaxResponse(Errors::IMPOSSIBLE);
                
                $bSearch = $this->input->postget('search', TYPE_BOOL);

                if ($bSearch) {
                    $aData['form'] = $this->dpForm($nCatID, true, array(), array('f'=>array()));
                } else {
                    $aData['form'] = $this->dpForm($nCatID, false, array());
                }

                $this->ajaxResponse($aData);
            } break;
            case 'item-img-upload':
            {
                $nItemID = $this->input->postget('id', TYPE_UINT);
                $aResponse = array();
                do{
                    if ($nItemID > 0)
                    {
                        $aData = $this->model->itemData($nItemID, array('user_id', 'img', 'imgcnt', 'status', 'moderated'));
                        if (empty($aData)) {
                            $this->errors->reloadPage();
                            break;
                        }

                        if($aData['status'] == self::STATUS_BLOCKED && $aData['moderated']==0) {
                            $this->errors->set(_t('','Объявление ожидает проверки модератора'));
                            break;
                        }

                        # автор объявления = загеристрированный пользователь
                        if( ! $nUserID # незарегистрированный пытается добавить фото к чужому объявлению
                            || ($nUserID > 0 && $aData['user_id'] != $nUserID) # текущий зарегистрированный пользователь не является владельцем объявления
                        ) {
                            $this->errors->reloadPage();
                            break;
                        }
                    } else {
                        # грузить новые фотографии(без привязки к объявлению) можно с ограничениями QQ
                    }

                    if( ! $this->security->validateReferer() ) {
                        $this->errors->reloadPage();
                        break;
                    }

                    $oImages = $this->initImages();
                    $aResult = $oImages->uploadImageQQ();
                    $aResponse['success'] = ($aResult !== false && $this->errors->no());
                    $aResponse['filename'] = $oImages->saveImageFileCustom($nItemID, $aResult);

                }while(false);

                $aResponse['errors'] = $this->errors->get(true);
                $this->ajaxResponse($aResponse, true, false, true);
            } break;
            case 'item-img-delete':
            {
                $nItemID = $this->input->post('id', TYPE_UINT); 

                do
                {
                    if($nItemID > 0)
                    {
                        $aData = $this->model->itemData($nItemID, array('user_id', 'img', 'imgcnt', 'status', 'moderated'));
                        if (empty($aData)) {
                            $this->errors->set( _t('', 'Редактируемое объявление не найдено'));
                            break;
                        }

                        if($aData['status'] == self::STATUS_BLOCKED && $aData['moderated']==0) {
                            $this->errors->set( _t('', 'Объявление ожидает проверки модератора'));
                            break;
                        }

                        # автор объявления = загеристрированный пользователь
                        if( ! $nUserID # незарегистрированный пытается удалить фото чужого объявлению
                        || ($nUserID > 0 && $aData['user_id'] != $nUserID) # текущий зарегистророванный пользователь не является владельцем объявления
                          ) {
                            $this->errors->set( _t('', 'Вы не является владельцем данного объявления.'));
                            break;
                        }
                    } else {
                        # удалять фотографии(без привязки к объявлению) можно без ограничений
                    }

                    if( ! $this->security->validateReferer() ) {
                        $this->errors->set( _t('', 'Ошибка удаления фотографии'));
                        break;
                    }

                    $aFilename = $this->input->post('filename', TYPE_ARRAY_NOTAGS);
                    if (empty($aFilename)) {
                        $this->errors->set( _t('', 'Ошибка удаления фотографии'));
                        break;
                    }

                    $oImages = $this->initImages();
                    foreach($aFilename as $file) {
                        $oImages->deleteImageFileCustom($nItemID, $file);
                    }
                } while(false);
                
                $this->ajaxResponseForm();
            } break;
            case 'item-del':
            {
                $nItemID = $this->input->post('id', TYPE_UINT);
                if( ! $nItemID || ! $nUserID) $this->ajaxResponse(Errors::IMPOSSIBLE);
                
                if( ! $this->security->validateToken()) {
                    $this->ajaxResponse(Errors::IMPOSSIBLE);
                }

                $res = $this->deleteItem($nItemID, $nUserID);
                $this->ajaxResponse( ($res ? Errors::SUCCESS : Errors::IMPOSSIBLE) );
                
            } break;              
            case 'item-publicate2':
            {
                $nItemID = $this->input->post('id', TYPE_UINT);
                $nPeriod = $this->input->post('period', TYPE_UINT);
                
                if( ! $nItemID || ! $nUserID || empty($nPeriod)) {
                    break;
                }
                if( ! $this->security->validateToken()) {
                    break;
                }

                $aData = $this->model->itemData($nItemID, array('id', 'user_id', 'status', 'publicated', 'publicated_to',
                        'moderated', 'cat_id', 'type_id'));
                if (empty($aData)) break;
                
                if( ! User::isCurrent($aData['user_id'])) {
                    break;
                }
                 
                if($aData['status'] == self::STATUS_BLOCKED) {
                    if($aData['moderated'] == 0) {
                        $this->errors->set( _t('realty', 'Невозможно продлить публикацию, поскольку объявление ожидает проверки') );
                    } else {
                        $this->errors->set( _t('realty', 'Невозможно продлить публикацию, поскольку объявление отклонено') );
                    }
                    break;
                } else if($aData['status'] == self::STATUS_PUBLICATED) {
                    $this->errors->set( _t('realty', 'Невозможно продлить публикацию, поскольку объявление опубликовано') );
                    break;
                } else if($aData['status'] != self::STATUS_PUBLICATED_OUT) {
                    $this->ajaxResponse(Errors::IMPOSSIBLE);
                }
            
                $publicateTo = $this->preparePublicatePeriodTo( $nPeriod, time() );
                if (empty($publicateTo)) {
                    $this->errors->set( _t('realty', 'Период публикации указан не корректно') );
                    break;
                }                    
            
                $toOld = strtotime( $aData['publicated_to'] );
                /* если разница между датой снятия с публикации и текущей датой
                 * более 3 дней, тогда поднимаем объявление вверх.
                 * в противном случае: оставлем дату старта публикации(pulicated) и дату порядка публикации(publicated_order) прежними
                 */
                $bUpdatePublicatedOrder = ((time() - $toOld) > 259200); //60*60*24*3
                $aUpdate = array(
                    'publicated_to' => $publicateTo,
                    'status_prev' => $aData['status'],
                    'status' => self::STATUS_PUBLICATED,
                );
                if ($bUpdatePublicatedOrder) {
                    $aUpdate['publicated'] = $this->db->now();
                    $aUpdate['publicated_order'] = $this->db->now();
                }

                $res = $this->model->itemSave($nItemID, $aUpdate);
                if( ! empty($res)) {
                    # накручиваем счетчики кол-ва опубликованных объявлений:
                    # в категории и типе категории:
                    $this->itemsCounterUpdate($aData['cat_id'], $aData['type_id'], true);
                    $this->ajaxResponse(Errors::SUCCESS);
                } else {
                    $this->ajaxResponse(Errors::IMPOSSIBLE);
                }
                
                $this->ajaxResponse( null );
                
            } break;
            case 'item-claim':
            {
                $p = $this->input->postm(array(
                    'id'      => TYPE_UINT,    
                    'reasons' => TYPE_ARRAY_UINT,  
                    'comment' => TYPE_NOTAGS,
                    'captcha' => TYPE_STR,
                ));
                
                $aResponse = array();
                do {
                    if( ! $p['id']) { $this->errors->impossible(); break; }
                    if (empty($p['reasons']) && $p['comment'] == '') { $this->errors->set( _t('realty', 'Укажите причину и/или описание') ); break; }
                    
                    if( ! $nUserID) {
                        if( ! $this->security->validateReferer()) {
                            $this->errors->impossible();
                            break;
                        }
                        $oProtection = new CCaptchaProtection();
                        if( ! $oProtection->valid($this->input->cookie('c2'), $p['captcha']) ) {
                            $aResponse['captcha'] = 1;
                            $this->errors->set( _t('', 'Результат с картинки указан некорректно'), 'captcha' );
                            break;               
                        }
                    } else {
                        if( ! $this->security->validateToken()) {
                            $this->errors->impossible();
                            break;
                        }
                        # получаем дату последней добавленной пользователем жалобы
                        $sLastClaim = $this->db->one_data('SELECT created FROM '.TABLE_REALTY_CLAIMS.' WHERE user_id = '.$nUserID.' ORDER BY created DESC LIMIT 1');
                        if( ! empty($sLastClaim)) {
                            $nMinutesTimeout = 3;
                            if( (time() - strtotime($sLastClaim)) < $nMinutesTimeout * 60 )
                            {
                                $this->errors->set( _t('realty', 'Повторите попытку через 3 минуты') );
                                break;
                            }
                        }
                    }
                    
                    $nReasons = array_sum($p['reasons']);
                    $res = $this->db->insert(TABLE_REALTY_CLAIMS, array(
                        'item_id' => $p['id'], 'user_id' => $nUserID, 'comment' => $p['comment'],
                        'reasons' => $nReasons, 'ip' => Request::remoteAddress(), 'created' => $this->db->now(),
                    ));
                    if($res) {
                        if( ! $nUserID) Request::deleteCOOKIE('c2');
                        config::saveCount('realty_claims', 1, true);
                    }
                    
                                        
                } while(false);

                $aResponse['res'] = $this->errors->no();
                $this->ajaxResponse($aResponse);                
            } break;
            case 'item-promote':
            {
                do
                {
                    $nItemID = $this->input->post('id', TYPE_UINT); # ID Объявления
                    $nSvcID = $this->input->post('svc', TYPE_UINT); # ID Услуги Realty::SERVICE_
                    $sPaySystem = $this->input->post('ps', TYPE_NOTAGS); # Ключ способа оплаты
                    if (empty($sPaySystem)) $sPaySystem = Bills::PS_BALANCE;

                    if ( ! $nItemID || ! $nSvcID || ! $nUserID || ! $this->security->validateToken()) {
                        $this->errors->reloadPage();
                        break;
                    }

                    $aItem = $this->model->itemData($nItemID, array('id','user_id','status'));
                    if (empty($aItem) || $aItem['status'] == self::STATUS_NEW) {
                        $this->errors->reloadPage();
                        break;
                    }
                    if ($aItem['status'] == self::STATUS_BLOCKED) {
                        $this->errors->set( _t('', 'Невозможно продвинуть заблокированное объявление') );
                        break;
                    }
                    if( ! User::isCurrent($aItem['user_id']) ) {
                        $this->errors->reloadPage();
                        break;
                    }

                    # проверяем способ оплаты
                    $aPaySystems = Bills::getPaySystems(true);
                    if (!isset($aPaySystems[$sPaySystem]) || empty($aPaySystems[$sPaySystem]['enabled'])) {
                        $this->errors->set( _t('bills', 'Выбранный способ оплаты на текущий момент недоступен') );
                        break;
                    }
                    # получаем стоимость услуги
                    $aSvc = Svc::model()->svcData($nSvcID);
                    if (empty($aSvc)) {
                        $this->errors->reloadPage();
                        break;
                    }
                    $nSvcPrice = $aSvc['price'];

                    $aPayAmount = Bills::getPayAmount($nSvcPrice, $sPaySystem);

                    $aResponse = $this->svc()->activateOrPay($this->module_name, $nSvcID, $nItemID,
                        $aPaySystems[$sPaySystem]['id'], $aPaySystems[$sPaySystem]['way'],
                        $nSvcPrice, $aPayAmount['amount'], $aPayAmount['currency']);
                    if ($aResponse['activated']) {
                        $aResponse['redirect'] = static::url('promote', array('success'=>1,'id'=>$nItemID,'svc'=>$nSvcID));
                    }

                } while(false);

                $this->ajaxResponseForm($aResponse);
            } break;
            case 'item-promote-data':
            {
                do
                {
                    $nItemID = $this->input->post('item_id', TYPE_UINT);

                    if( ! $nItemID || ! $nUserID ) {
                        $this->errors->reloadPage();
                        break;
                    }

                    $aItem = $this->model->itemData($nItemID, array('id','user_id','status',
                             'svc', 'marked_to', 'fixed_to', 'vip_to'));
                    if( empty($aItem)
                        || $aItem['status'] == self::STATUS_NEW
                        || $aItem['status'] == self::STATUS_BLOCKED
                        || ! User::isCurrent( $aItem['user_id'] )
                        || ! $this->security->validateReferer() ) {
                        $this->errors->reloadPage();
                        break;
                    }
                    $aResponse['marked'] = ($aItem['svc'] & self::SERVICE_MARK);
                    $aResponse['marked_to'] = _t('svc', 'Услуга уже активирована до <b>[date]</b>', array('date'=>tpl::date_format2($aItem['marked_to'], false)));
                    $aResponse['fixed'] = ($aItem['svc'] & self::SERVICE_FIX);
                    $aResponse['fixed_to'] = _t('svc', 'Услуга уже активирована до <b>[date]</b>', array('date'=>tpl::date_format2($aItem['fixed_to'], false)));
                    $aResponse['vip'] = ($aItem['svc'] & self::SERVICE_VIP);
                    $aResponse['vip_to'] = _t('svc', 'Услуга уже активирована до <b>[date]</b>', array('date'=>tpl::date_format2($aItem['vip_to'], false)));
                } while(false);

                $this->ajaxResponseForm($aResponse);
            } break;
        }

        $this->ajaxResponse(Errors::IMPOSSIBLE);
    }

    /**
     * Формируем правый блок
     * @param string $sPage ключ страницы, на которой формируем
     * @param array $aData доп. данные
     */
    protected function initRightblock($sPage, array $aData = array())
    {
        $bAfterBanner = false;

        switch ($sPage)
        {
            case 'index':
            case 'search':
            {

            } break;
        }
        bff::setRightblock($this->viewPHP($aData, 'common.rightblock'), $bAfterBanner);
    }

    /**
     * Формируем VIP блок
     * @param int $nCatID ID категории
     * @param int $nTypeID ID типа категории
     * @param int $nCityID ID города
     * @return string
     */
    protected function vipBlock($nCatID = 0, $nTypeID = 0, $nCityID = 0)
    {
        if (!$nCityID && $this->regionsFilterEnabled() && ! Geo::cityIsOne()) {
            $nCityID = Geo::cityID();
        }
        $aData['items'] = $this->model->itemsListVIP($nCatID, $nTypeID, $nCityID);
        if ( ! empty($aData['items']) ) {
            foreach ($aData['items'] as &$v) {
                $v['img'] = RealtyImages::url($v['id'], $v['imgfav'], RealtyImages::szThumbnail);
            } unset($v);
        }
        return $this->viewPHP($aData, 'vip.block');
    }

    /**
     * Cron задачи модуля
     * рекомендуемая периодичность: раз в час
     */
    public function cron()
    {
        if ( ! bff::cron()) return;


        # снимаем с публикации объявления, у которых закончился срок публикации
        $this->db->exec('UPDATE '.TABLE_REALTY_ITEMS.'
                       SET status_prev = status, status = :status_new
                       WHERE status = :status_prev AND publicated_to <= :now',
                       array(
                            ':status_new' => self::STATUS_PUBLICATED_OUT,
                            ':status_prev' => self::STATUS_PUBLICATED,
                            ':now' => $this->db->now(),
                       ));

        # формируем YandexXML
        $this->yandexXML();

        {
            # пересчитываем счетчики объявлений
            
            # Категории
            $this->db->exec('UPDATE '.TABLE_REALTY_CATEGORIES.' C
                   LEFT JOIN (SELECT I.cat_id as id, COUNT(I.id) as items
                                FROM '.TABLE_REALTY_ITEMS.' I, '.TABLE_USERS.' U
                                WHERE I.status = :status AND I.user_id = U.user_id AND U.blocked = 0
                                '.(static::premoderation() ? ' AND I.moderated > 0' : '').'
                                GROUP BY I.cat_id) as X ON C.id = X.id
                    SET C.items = IF(X.id IS NOT NULL, X.items, 0)
            ', array(':status'=>self::STATUS_PUBLICATED));

            # Типы категорий
            $this->db->exec('UPDATE '.TABLE_REALTY_CATEGORIES_TYPES.' C
                   LEFT JOIN (SELECT I.cat_id, I.type_id, COUNT(I.id) as items
                                FROM '.TABLE_REALTY_ITEMS.' I, '.TABLE_USERS.' U
                                WHERE I.status = :status AND I.user_id = U.user_id AND U.blocked = 0
                                '.(static::premoderation() ? ' AND I.moderated > 0' : '').'
                                GROUP BY I.cat_id, I.type_id) as X ON C.cat_id = X.cat_id AND C.type_id = X.type_id
                    SET C.items = IF(X.cat_id IS NOT NULL, X.items, 0)
            ', array(':status'=>self::STATUS_PUBLICATED));
        }
    }
    
}