<?php

use bff\utils\Files;

define('TABLE_JOB_CATEGORIES',      DB_PREFIX.'job_categories');
define('TABLE_JOB_CATEGORIES_LANG', DB_PREFIX.'job_categories_lang');
define('TABLE_JOB_RESUME',          DB_PREFIX.'job_resume');
define('TABLE_JOB_VACANCY',         DB_PREFIX.'job_vacancy');
define('TABLE_JOB_CLAIMS',          DB_PREFIX.'job_claims');
define('TABLE_JOB_FAV',             DB_PREFIX.'job_fav');

abstract class JobBase extends Module implements IModuleWithSvc
{
    /** @var JobModel */
    public $model = null;
    protected $securityKey = 'b43b2028e79732110516de8a2f7532fd';

    const STATUS_NEW            = 1; //новое
    const STATUS_PUBLICATED     = 3; //опубликованное
    const STATUS_PUBLICATED_OUT = 4; //истекший срок публикации
    const STATUS_BLOCKED        = 5; //заблокированное

    const TYPE_VACANCY = 1;
    const TYPE_RESUME  = 2;

    const resumeYears = 'FLOOR(DATEDIFF(NOW(), I.birthdate)/365) as years';

    # ID Услуг
    const SERVICE_RESUME_UP    = 8192;   # резюме: поднятие
    const SERVICE_RESUME_MARK  = 16384;  # резюме: выделенние
    const SERVICE_RESUME_FIX   = 32768;  # резюме: закрепление
    const SERVICE_VACANCY_UP   = 65536;  # вакансии: поднятие
    const SERVICE_VACANCY_MARK = 131072; # вакансии: выделенние
    const SERVICE_VACANCY_FIX  = 262144; # вакансии: закрепление

    public function init()
    {
        parent::init();

        $this->module_title = 'Работа';
    }

    /**
     * @return Job
     */
    public static function i()
    {
        return bff::module('job');
    }

    /**
     * @return JobModel
     */
    public static function model()
    {
        return bff::model('job');
    }

    /**
     * Формирование URL
     * @param string $key ключ
     * @param mixed $opts параметры
     * @param boolean $dynamic динамическая ссылка
     * @return string
     */
    public static function url($key = '', $opts = array(), $dynamic = false)
    {
        $base = static::urlBase(LNG, $dynamic);
        switch ($key)
        {
            # Просмотр вакансии / резюме
            case 'view':
                return strtr($opts, array(
                    '{sitehost}' => SITEHOST . bff::locale()->getLanguageUrlPrefix(),
                ));
                break;

            # Вакансии: список (поиск)
            case 'vacancy.list':
                return $base.'/vacancy/'.static::urlQuery($opts);
                break;
            # Вакансии: добавление
            case 'vacancy.add':
                return $base.'/vacancy/add'.static::urlQuery($opts);
                break;
            # Вакансии: редактирование
            case 'vacancy.edit':
                return $base.'/vacancy/edit'.static::urlQuery($opts);
                break;

            # Резюме: список (поиск)
            case 'resume.list':
                return $base.'/resume/'.static::urlQuery($opts);
                break;
            # Резюме: добавление
            case 'resume.add':
                return $base.'/resume/add'.static::urlQuery($opts);
                break;
            # Резюме: редактирование
            case 'resume.edit':
                return $base.'/resume/edit'.static::urlQuery($opts);
                break;

            # Компании: список
            case 'company.list':
                return $base.'/company/'.static::urlQuery($opts);
                break;

            # Агентства
            case 'agents.list':
                return $base.'/agents'.static::urlQuery($opts);
                break;

            # Продвижение
            case 'promote':
                return $base.'/promote'.static::urlQuery($opts);
                break;

            # Фильтр
            case 'search':
                if (!empty($opts['type']) && is_int($opts['type'])) {
                    switch ($opts['type']) {
                        case self::TYPE_VACANCY: $opts['type'] = 'vacancy'; break;
                        case self::TYPE_RESUME:  $opts['type'] = 'resume';  break;
                    }
                }
                return $base.
                    ( ! empty($opts['type']) ? '/'.$opts['type'] : '').
                    '/search/'.
                    ( ! empty($opts['cat']) ? $opts['cat'] : '' ).
                static::urlQuery($opts, array('type', 'cat'));
                break;

            # Главная
            case 'index':
                return $base.'/'.static::urlQuery($opts);
                break;
        }
        return $base;
    }

    public static function urlViewVacancy($id, $keyword = '', $cityID = 0)
    {
        if (empty($keyword)) $keyword = 'item';
        return static::urlBase(LNG, true, array('city'=>$cityID)).'/vacancy/'.$keyword.'-'.$id.'.html';
    }

    public static function urlViewResume($id, $keyword = '', $cityID = 0)
    {
        if (empty($keyword)) $keyword = 'item';
        return static::urlBase(LNG, true, array('city'=>$cityID)).'/resume/'.$keyword.'-'.$id.'.html';
    }

    /**
     * Включена ли премодерация
     * @return boolean
     */
    public static function premoderation()
    {
        return (bool)config::sys('job.premoderation', true);
    }

    /**
     * ID категории агенств
     * @return int
     */
    public static function agentsCategory()
    {
        return intval(config::sys('job.agents.category', 134));
    }

    /**
     * Описание seo шаблонов страниц
     * @return array
     */
    public function seoTemplates()
    {
        $aTemplates = array(
            'groups' => array(
                'def' => array('t'=>'Общие'),
                self::TYPE_VACANCY => array('t'=>'Вакансии', 'm'=>array('вакансии','вакансий')),
                self::TYPE_RESUME  => array('t'=>'Резюме', 'm'=>array('резюме','резюме')),
            ),
            'pages' => array(
                'index' => array( // index
                    't'      => 'Главная страница',
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'company' => array( // agents
                    't'      => 'Компании - список',
                    'list'   => true,
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'agents' => array( // agents
                    't'      => 'Агентства - список',
                    'list'   => true,
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
            ),
        );

        foreach (array(
            self::TYPE_VACANCY => $aTemplates['groups'][self::TYPE_VACANCY],
            self::TYPE_RESUME  => $aTemplates['groups'][self::TYPE_RESUME]
        ) as $k=>$v) {
            $aTemplates['pages']['index-'.$k] = array( // vacancy_index, resume_index
                't'      => 'Банк '.$v['m'][1].' (главная)',
                'group'  => $k,
                'macros' => array(
                    'region' => array('t' => 'Регион'),
                ),
                'fields' => array(
                    'titleh1' => array(
                        't'      => 'Заголовок H1',
                        'type'   => 'text',
                    ),
                ),
            );
            $aTemplates['pages']['search-'.$k] = array( // search
                't'      => 'Поиск '.$v['m'][1],
                'list'   => true,
                'group'  => $k,
                'macros' => array(
                    'region' => array('t' => 'Регион'),
                ),
                'fields' => array(
                    'titleh1' => array(
                        't'      => 'Заголовок H1',
                        'type'   => 'text',
                    ),
                ),
            );
            $aTemplates['pages']['search-category-'.$k] = array( // search
                't'      => 'Поиск '.$v['m'][1].' (категория)',
                'list'   => true,
                'inherit'=> true,
                'group'  => $k,
                'macros' => array(
                    'category' => array('t' => 'Название сферы деятельности'),
                    'region' => array('t' => 'Регион'),
                ),
                'fields' => array(
                    'titleh1' => array(
                        't'      => 'Заголовок H1',
                        'type'   => 'text',
                    ),
                ),
            );
            $aTemplates['pages']['add-'.$k] = array( // vacancy_add, resume_add
                't'      => 'Добавление '.$v['m'][0],
                'group'  => $k,
                'macros' => array(
                    'region' => array('t' => 'Регион'),
                ),
                'fields' => array(
                    // ...
                ),
            );
            $viewMacros = array();
            if ($k == self::TYPE_VACANCY) {
                $viewMacros['company'] = array('t' => 'Название компании');
            } else if ($k == self::TYPE_RESUME) {
                // ...
            }
            $viewMacros['region'] = array('t' => 'Регион');
            $aTemplates['pages']['view-'.$k] = array( // vacancy_view, resume_view
                't'      => 'Просмотр '.$v['m'][0],
                'i'      => true,
                'group'  => $k,
                'macros' => array_merge(array(
                    'title'  => array('t' => 'Заголовок '.$v['m'][0]),
                    'description'  => array('t' => 'Описание (до 150 символов)'),
                ), $viewMacros),
                'fields' => array(
                    'share_title'       => array(
                        't'    => 'Заголовок (поделиться в соц. сетях)',
                        'type' => 'text',
                    ),
                    'share_description' => array(
                        't'    => 'Описание (поделиться в соц. сетях)',
                        'type' => 'textarea',
                    ),
                    'share_sitename'    => array(
                        't'    => 'Название сайта (поделиться в соц. сетях)',
                        'type' => 'text',
                    ),
                ),
            );
        }

        return $aTemplates;
    }

    /**
     * Актуализация счетчика вакансий/резюме ожидающих модерации
     * @param integer $typeID тип объявления
     * @param null $increment
     */
    function moderationCounter($typeID, $increment = null)
    {
        $counterKey = 'job_'.($typeID == self::TYPE_VACANCY?'vacancy':'resume').'_mod';
        if (empty($increment)) {
            $count = $this->model->itemsModerationCounter($typeID);
            bff::log('job mod ('.$typeID.') = '.$count);
            config::save($counterKey, $count, true);
        } else {
            config::saveCount($counterKey, $increment, true);
        }
    }

    function getSchedule($nSelected = 0, $htmlType = 'radio', $sFieldname = 'schedule')
    {
        $aTypes = array(
            1  => _t('job', 'Полный рабочий день'),
            2  => _t('job', 'Стажировка/практика'),
            4  => _t('job', 'Сезонная/временная работа'),
            8  => _t('job', 'Частичная занятость'),
            16 => _t('job', 'Удаленная работа'),
            32 => _t('job', 'Проектная работа'),
        );
        
        $html = '';
        if($htmlType === true) { // для фильтра
            $i = 0;
            foreach($aTypes as $k=>$v) 
            {
                $html .= '<li><label class="checkbox"><input type="checkbox" name="'.$sFieldname.'['.($i++).']" value="'.$k.'" '.($k & $nSelected ? ' checked="checked"':'').' class="radio" /> '.$v.'</label></li>';
            }
            return $html;
        } elseif($htmlType === 'radio') {
            foreach($aTypes as $k=>$v) 
            {
                $html .= '<p><label class="radio"><input type="radio" name="'.$sFieldname.'" value="'.$k.'" '.($k == $nSelected ? ' checked="checked"':'').' class="checkbox" /> '.$v.'</label></p>';
            }
            return $html;
        } elseif($htmlType === 'checkbox') {
            foreach($aTypes as $k=>$v) 
            {
                $html .= '<p><label class="checkbox"><input type="checkbox" name="'.$sFieldname.'" value="'.$k.'" '.($k & $nSelected ? ' checked="checked"':'').' class="checkbox" /> '.$v.'</label></p>';
            }
            return $html;
        } elseif($htmlType === 1) { // для просмотра
            return ( isset($aTypes[$nSelected]) ? $aTypes[$nSelected] : '' );
        } else {
            return $aTypes;
        }
    }

    function getExperience($nSelected = 0, $htmlType = 'options')
    {
        $aTypes = array(
            1  => _t('job', 'нет опыта работы'),
            2  => _t('job', 'до 2 лет'),
            3  => _t('job', '2-3 года'),
            4  => _t('job', '3-5 лет'),
            5  => _t('job', 'более 5 лет'),
        );
        
        if($htmlType === 'options') {
            $html = '';
            if (empty($nSelected)) {
                $html .= '<option value="0">'._t('', 'выберите из списка').'</option>';
            }
            foreach($aTypes as $k=>$v) {
                $html .= '<option value="'.$k.'" '.($k == $nSelected ? ' selected="selected"':'').'>'.$v.'</option>';
            }
            return $html;
        } elseif($htmlType === 1) { // для просмотра
            return ( isset($aTypes[$nSelected]) ? $aTypes[$nSelected] : '' );
        } else {
            return $aTypes;
        }
    }

    function getEducation($nSelected = 0, $htmlType = 'radio', $sFieldname = 'education')
    {
        $aTypes = array(
            1  => _t('job', 'высшее'),
            2  => _t('job', 'неполное высшее'),
            3  => _t('job', 'среднее специальное'),
            4  => _t('job', 'среднее'),
        );
        
        if($htmlType === 'radio') {
            $html = '';
            foreach($aTypes as $k=>$v) {
                $html .= '<p><label class="radio"><input type="radio" name="'.$sFieldname.'" value="'.$k.'" '.($k == $nSelected ? ' checked="checked"':'').' class="checkbox" /> '.$v.'</label></p>';
            }
            return $html;
        } elseif($htmlType === 1) { // для просмотра
            return ( isset($aTypes[$nSelected]) ? $aTypes[$nSelected] : '' );
        } else {
            return $aTypes;
        }
    }

    function getPCLevel($nSelected = 0, $htmlType = 'radio', $sFieldname = 'pclevel')
    {
        $aTypes = array(
            1  => _t('job', 'базовый'),
            2  => _t('job', 'пользователь'),
            3  => _t('job', 'опытный пользователь'),
            4  => _t('job', 'профи'),
        );
        
        if($htmlType === 'radio') {
            $html = '';
            foreach($aTypes as $k=>$v) {
                $html .= '<p><label class="radio"><input type="radio" name="'.$sFieldname.'" value="'.$k.'" '.($k == $nSelected ? ' checked="checked"':'').' class="checkbox" /> '.$v.'</label></p>';
            }
            return $html;
        } elseif($htmlType === 1) { // для просмотра
            return ( isset($aTypes[$nSelected]) ? $aTypes[$nSelected] : '' );
        } else {
            return $aTypes;
        }
    }

    function getPublicatedTo($sType = 'vacancy')
    {
        return date('Y-m-d H:i:s', strtotime('+2 months'));
    }

    function getPublicatePeriods() //admin: publicate2
    {
        $periods = array();
        $week = 604800;
        $from = time();
        $periods[1] = date('d.m.Y', $from + $week);     //1 неделя
        $periods[2] = date('d.m.Y', $from + ($week*2)); //2 неделя
        $periods[3] = date('d.m.Y', $from + ($week*3)); //3 неделя
        $nM = (int)date('m', $from);
        $nD = date('d', $from);
        $nY = date('Y', $from);
        $periods[4] = date('d.m.Y', mktime(0,0,0,$nM+1,$nD,$nY)); //1 месяц
        $periods[5] = date('d.m.Y', mktime(0,0,0,$nM+2,$nD,$nY)); //2 месяца
      //  $periods[6] = date('d.m.Y', mktime(0,0,0,$nM+3,$nD,$nY)); //3 месяца
      //  $periods[9] = date('d.m.Y', mktime(0,0,0,$nM+6,$nD,$nY)); //6 месяцев
        
        $options = '<option value="1" selected="selected">'._t('', 'на 1 неделю').'</option>
                    <option value="2">'._t('', 'на 2 недели').'</option>
                    <option value="3">'._t('', 'на 3 недели').'</option>
                    <option value="4">'._t('', 'на 1 месяц').'</option>
                    <option value="5">'._t('', 'на 2 месяца').'</option>';
//                    <option value="6">на 3 месяца</option>
//                    <option value="9">на 6 месяцев</option>';
        
        return array('dates'=>$periods, 'options'=>$options);
    }

    function preparePublicatePeriodTo($nPeriod, $nFrom) //admin: publicate2
    {
        $dateTo = false;
        $dateFormat = 'Y-m-d H:i:s';
        $dateWeek = (60*60 * 24 * 7);
        switch ( $nPeriod )
        {
            case 2: $dateTo = date($dateFormat, $nFrom+$dateWeek*2);  break; //2 недели
            case 3: $dateTo = date($dateFormat, $nFrom+$dateWeek*3);  break; //3 недели
            case 4: //1 месяц
            case 5: //2 месяца
           // case 6: //3 месяца
           // case 9: //6 месяцев
            {
                $dateTo = date($dateFormat, mktime(date('H', $nFrom),date('i', $nFrom),date('s', $nFrom),date('m', $nFrom)+($nPeriod-3),date('d', $nFrom),date('Y', $nFrom)) );
            } break;
            case 1: default: $dateTo = date($dateFormat, $nFrom+$dateWeek);  break;   //1 неделя
        }
        return $dateTo;
    }

    function itemsCounterUpdate($nCatID, $sType, $bIncrement)
    {
        $sqlField = ($sType == 'vacancy' ? 'vacancy_items' : 'resume_items');
        
        $this->db->exec('UPDATE '.TABLE_JOB_CATEGORIES.'
            SET '.$sqlField.' = '.$sqlField.' '.($bIncrement?'+ 1':'- 1').'
            WHERE id = '.$nCatID);
    }

    # ----------------------------------------------------------------------------------------------------
    # вакансии

    /**
     * Обрабатываем параметры вакансии
     */
    function processVacancyData($bNew, &$aFields = array())
    {
        $aParams = array(
            # Информация о вакансии:
            'title'        => array(TYPE_NOTAGS, 'len'=>200), // название должности
            'description'  => array(TYPE_NOTAGS, 'len'=>10000), // описание
            'city_id'      => TYPE_UINT,    // город
            'cat_id'       => TYPE_UINT,    // сфера деятельности
            'schedule'     => TYPE_UINT,    // график работы
            'price'        => TYPE_PRICE,   // зарплата, от
            'price_torg'   => TYPE_BOOL,    // зарплата: параметры; по договоренности...
            'price_curr'   => TYPE_UINT,    // зарплата: валюта
            # Информация о работодателе:
            'company_id'          => TYPE_UINT,    // компания
            'company_title'       => array(TYPE_NOTAGS, 'len'=>200),  // другая (название)
            'company_staffagency' => TYPE_BOOL,    // кадровое агенство
            'company_cityid'      => TYPE_UINT,    // город 
            'company_addr'        => array(TYPE_NOTAGS, 'len'=>250),  // адрес
            'company_phone'       => array(TYPE_NOTAGS, 'len'=>100),  // телефон
            'company_email'       => array(TYPE_NOTAGS, 'len'=>100),  // email
            'company_contact'     => array(TYPE_NOTAGS, 'len'=>150),  // контактное лицо
        );

        $aData = $this->input->postm($aParams);
        if (Request::isPOST())
        {
            if (strlen($aData['title'])<2) {
                $this->errors->set( _t('job', 'Укажите название должности'), 'title' );
                $aFields[] = 'title';
            }
            $aData['keyword'] = mb_strtolower(func::translit($aData['title']));
            if (strlen($aData['description'])<2) {
                $this->errors->set( _t('job', 'Укажите описание вакансии'), 'description' );
            }
            if( ! Geo::cityIsValid($aData['city_id']) ) {
                $this->errors->set( _t('job', 'Выберите город из представленных в списке'), 'city_id' );
            }
            if( ! $aData['cat_id'] ) {
                $this->errors->set( _t('job', 'Укажите сферу деятельности'), 'cat_id' );
            }
            if( ! $aData['price'] ) {
                $this->errors->set( _t('job', 'Укажите минимальную зарплату'), 'price' );
                $aFields[] = 'price';
            }
            if( ! ($aData['company_id']) && strlen($aData['company_title'])<5 ) {
                $this->errors->set( _t('job', 'Укажите название компании'), 'company_title' );
                $aFields[] = 'company_title';
            }
            if( ! Geo::cityIsValid($aData['company_cityid']) ) {
                $this->errors->set( _t('job', 'Выберите город компании из представленных в списке') );
            }
            if( empty($aData['company_phone']) || strlen($aData['company_phone'])<5 ) {
                $this->errors->set( _t('job', 'Укажите корректный номер телефона'), 'company_phone' );
                $aFields[] = 'company_phone';
            }
            if( ! $this->input->isEmail($aData['company_email'], false) ) {
                $this->errors->set( _t('job', 'Укажите корректный e-mail адрес'), 'company_email' );
                $aFields[] = 'company_email';
            }
            if( empty($aData['company_contact']) || strlen($aData['company_contact'])<2 ) {
                $this->errors->set( _t('job', 'Укажите контактное лицо'), 'company_contact' );
                $aFields[] = 'company_contact';
            }

        }
        return $aData;
    }

    function vacancyDelete($nItemID)
    {
        if ($nItemID<=0) return false;
        
        $aData = $this->db->select_row(TABLE_JOB_VACANCY, array('*'), array('id'=>$nItemID));
        if (empty($aData)) return false;

        $res = $this->db->delete(TABLE_JOB_VACANCY, array('id'=>$nItemID));
        if ($res) {
            $this->db->delete(TABLE_JOB_FAV, array('item_id'=>$nItemID, 'type_id'=>self::TYPE_VACANCY));
            $this->db->delete(TABLE_JOB_CLAIMS, array('item_id'=>$nItemID, 'type_id'=>self::TYPE_VACANCY));
            if (static::premoderation() && ! $aData['moderated']) {
                $this->moderationCounter(self::TYPE_VACANCY);
            }
        }
        
        return !empty($res);
    }

    public static function vacancyListing($aData, $sType, $width1, $width2 = 135)
    {
        $bFav = ($sType == 'fav');
        foreach($aData as $v): ?>
        <tr class="item job<? if($v['marked']) { ?> svc-item-marked<? } elseif($v['fixed']) { ?> fixed<? } ?>">
            <td class="content">
                <a href="<?= Job::url('view', $v['link']) ?>"><?= $v['title'] ?></a>
                <div class="f12 mt5">
                    <span class="grey"><?= $v['cat_title'] ?></span><br />
                    <span class="grey"><?= _t('', 'г.'); ?><?= $v['city_title'] ?></span><br/>
                    <span><?= tpl::date_format_spent($v['created'], false) ?></span>
                </div>
            </td>
            <td class="price">
                <div class="green"><b><?= ($v['price_torg'] ? _t('job','договорная') : _t('job','от').' '.tpl::formatPrice($v['price'], $v['price_curr'])) ?></b></div>
                <? if($v['company_id']>0){ ?><div class="f12 mt5 mb10">&laquo;<a href="<?= Items::url('view', $v['item_link']); ?>"><?= $v['item_title']; ?></a>&raquo;</div><? } else { ?><div class="f12 mt5 mb10">&laquo;<?= $v['company_title'];?>&raquo;</div><? } ?>

                <? if($bFav) { ?><div>
                    <a href="#" onclick="return app.fav('job',this, $(this).parent().parent());" data-id="<?= $v['id'] ?>" data-type="<?= Job::TYPE_VACANCY ?>" class="infav" title="<?= _t('', 'Удалить'); ?>"></a>
                    </div><? }
                else { ?> <div> <?
                    if(User::id()) { ?>
                        <a href="#" onclick="return app.fav('job',this);" data-id="<?= $v['id'] ?>" data-type="<?= Job::TYPE_VACANCY ?>"
                        <? if(!$v['fav']){ ?>class="intofav" title="<?= _t('', 'В избранное'); ?>" <? } else { ?>class="infav" title="<?= _t('', 'Избранное'); ?>" <? } ?>></a>
                </div><? } ?>
                <? } ?>
            </td>
        </tr>
        <?
        endforeach;
    }

    # ----------------------------------------------------------------------------------------------------
    # резюме

    /**
    * Обрабатываем параметры резюме
    */
    function processResumeData($bNew, &$aFields = array())
    {
        $aParams = array(
            //Пожелания к работе:
            'title'        => array(TYPE_NOTAGS, 'len'=>200),  // название должности
            'description'  => array(TYPE_NOTAGS, 'len'=>10000),  // профессиональные навыки
            'experience'   => TYPE_UINT,    // опыт работы в данной должности
            'cat_id'       => TYPE_UINT,    // профессиональная сфера
            'price'        => TYPE_PRICE,   // зарплата(в месяц), от
            'price_torg'   => TYPE_BOOL,    // зарплата: параметры; по договоренности...
            'price_curr'   => TYPE_UINT,    // зарплата: валюта  
            'schedule'     => TYPE_UINT,    // график работы 
            'city_id'      => TYPE_UINT,    // город, где я живу
            'city_searchwork'  => TYPE_BOOL,    //  ищу работу в этом городе
            'city_readytomove' => TYPE_BOOL,    //  готов к перезду
 
            //Образование и другие навыки:
            'education'    => TYPE_UINT,    // основное образование
            'pclevel'      => TYPE_UINT,    // знание ПК (уровень)
            
            //Последнее место работы:
            'company_title' => array(TYPE_NOTAGS, 'len'=>200),  // название компании
            'company_post'  => array(TYPE_NOTAGS, 'len'=>200),  // должность
            'company_duty'  => array(TYPE_NOTAGS, 'len'=>10000),  // служебные обязанности
            'company_from'  => TYPE_ARRAY_UINT,  // период работы, с
            'company_to'    => TYPE_ARRAY_UINT,  // период работы, по
            
            //Личные сведения:
            'fio'           => TYPE_NOTAGS,       // фамилия, имя, отчество
            'birthdate'     => TYPE_ARRAY_UINT,   // дата рождения
            'sex'           => TYPE_UINT,         // пол, 2 - м, 1 - ж
            'phones'        => TYPE_ARRAY_NOTAGS, // телефоны
            'email'         => TYPE_NOTAGS,       // email 
        );

        $aData = $this->input->postm($aParams);
        if (Request::isPOST())
        {
            // Пожелания к работе:
            if( strlen($aData['title'])<2 ) {
                $this->errors->set( _t('job', 'Укажите название должности'), 'title' );
                $aFields[] = 'title';
            }
            $aData['keyword'] = mb_strtolower(func::translit($aData['title']));
            if( strlen($aData['description'])<2 ) {
                $this->errors->set( _t('job', 'Опишите свои знания и навыки'), 'description' );
            }
            if( ! $aData['experience'] ) {
                $this->errors->set( _t('job', 'Укажите ваш опыт работы'), 'experience' );
            }
            if( ! $aData['cat_id'] ) {
                $this->errors->set( _t('job', 'Укажите профессиональную сферу'), 'cat_id' );
            }
            if( ! $aData['price'] ) {
                $this->errors->set( _t('job', 'Укажите минимальную зарплату в месяц'), 'price' );
                $aFields[] = 'price';
            }
            if( ! $aData['schedule'] ) {
                $this->errors->set( _t('job', 'Укажите график работы'), 'schedule' );
            }
            if( ! Geo::cityIsValid($aData['city_id']) ) {
                $this->errors->set( _t('job', 'Выберите город из представленных в списке'), 'city_id' );
            }
            
            // Период работы
                $bWrongPeriod = false;
                $this->input->clean_array($aData['company_from'], array(
                    'day'   => TYPE_UINT,
                    'month' => TYPE_UINT,
                    'year'  => TYPE_UINT,
                ));
                if( ! checkdate($aData['company_from']['month'], $aData['company_from']['day'], $aData['company_from']['year'])){
                    $bWrongPeriod = true;
                    $this->errors->set( _t('job', 'Период работы указан некорректно') );
                } else {
                    $aData['company_from'] = "{$aData['company_from']['year']}-{$aData['company_from']['month']}-{$aData['company_from']['day']}";
                }

                $this->input->clean_array($aData['company_to'], array(
                    'day'   => TYPE_UINT,
                    'month' => TYPE_UINT,
                    'year'  => TYPE_UINT,
                ));
                if( ! checkdate($aData['company_to']['month'], $aData['company_to']['day'], $aData['company_to']['year'])){
                    if(!$bWrongPeriod) {
                        $this->errors->set( _t('job', 'Период работы указан некорректно') );
                        $bWrongPeriod = true;
                    }
                } else {
                    $aData['company_to'] = "{$aData['company_to']['year']}-{$aData['company_to']['month']}-{$aData['company_to']['day']}";
                }
            
            // Личные сведения: 
            
            if( strlen($aData['fio'])<5 ) {
                $this->errors->set( _t('job', 'Укажите ваше ФИО'), 'fio' );
                $aFields[] = 'fio';
            }
            
            // - Дата рождения
                $this->input->clean_array($aData['birthdate'], array(
                    'day'   => TYPE_UINT,
                    'month' => TYPE_UINT,
                    'year'  => TYPE_UINT,
                ));
                if(!checkdate($aData['birthdate']['month'], $aData['birthdate']['day'], $aData['birthdate']['year'])){
                    $this->errors->set( _t('job', 'Дата рождения указана некорректно') );
                } else {
                    $aData['birthdate'] = "{$aData['birthdate']['year']}-{$aData['birthdate']['month']}-{$aData['birthdate']['day']}";
                }
            
            // - Контактные телефоны
                $aTemp = array(); $limit = 3;
                foreach($aData['phones'] as $p) {
                    $p = preg_replace('/[^\(\)\+\-0-9 ]/','',trim($p));
                    $p = preg_replace('/\s+/', ' ', $p);
                    if(strlen($p)>4) { $aTemp[] = mb_substr($p, 0, 40); }
                }
                if($limit > 0 && sizeof($aTemp) > $limit) $aTemp = array_slice($aTemp, 0, $limit);
                if (empty($aTemp)) { // минимум 1 телефон
                    $this->errors->set( _t('job', 'Укажите контактный телефон') );
                }
                $aData['phones'] = serialize($aTemp);
        }
        else
        {
//            $aData['phones']  = (!empty($aData['phones']) && is_string($aData['phones']) ? unserialize($aData['phones']) : array() );
//            if(!empty($aData['phones'])) { $phones = array(); foreach($aData['phones'] as $v){ $phones[] = $v['v']; } $aData['phones'] = $phones; }
        }
        return $aData;
    }

    function resumeDelete($nItemID)
    {
        if($nItemID<=0) return false;
        
        $aData = $this->db->one_array('SELECT * FROM '.TABLE_JOB_RESUME.' WHERE id = '.$nItemID);
        if (empty($aData)) return false;
        
        $res = $this->db->exec('DELETE FROM '.TABLE_JOB_RESUME.' WHERE id = '.$nItemID);
        if($res) {
            $this->db->exec('DELETE FROM '.TABLE_JOB_FAV.' WHERE item_id = '.$nItemID.' AND type_id = '.self::TYPE_RESUME);
            $this->db->exec('DELETE FROM '.TABLE_JOB_CLAIMS.' WHERE item_id = '.$nItemID.' AND type_id = '.self::TYPE_RESUME);
            if (static::premoderation() && ! $aData['moderated']) {
                $this->moderationCounter(self::TYPE_RESUME);
            }
        }
        return !empty($res);
    }

    function resumePhonesPrepareEdit($mPhones)
    {
        $aPhones = (is_string($mPhones) ? @unserialize($mPhones) : array() );
        if($aPhones === false) $aPhones = array();
        return $aPhones;
    }

    static function resumeListing($aData, $sType, $width1, $width2 = 135)
    {
        $bFav = ($sType == 'fav');
        foreach($aData as $v) { ?>
        <tr class="item job<? if($v['marked']) { ?> svc-item-marked<? } elseif($v['fixed']) { ?> fixed<? } ?>">
            <td class="content">
                <a href="<?= Job::url('view', $v['link']) ?>"><?= $v['title'] ?></a>
                <div class="f12 mt5">
                    <span class="grey"><?= $v['cat_title'] ?></span><br/>
                    <span class="grey"><?= _t('', 'г.'); ?><?= $v['city_title'] ?></span><br/>
                    <span><?= tpl::date_format_spent($v['created'], false) ?></span>
                </div>
            </td>
            <td class="price">
                <div class="green"><b><?= ($v['price_torg'] ? _t('job','договорная') : _t('job','от').' '.tpl::formatPrice($v['price'], $v['price_curr'])) ?></b></div>
                <div class="f12 mt5 mb10 nowrap"><?= $v['fio']; ?>, <?= tpl::declension($v['years'], _t('','год;года;лет')) ?></div>
                <? if($bFav) { ?><div>
                    <a href="#" onclick="return app.fav('job',this, $(this).parent().parent().parent());" data-id="<?= $v['id'] ?>" data-type="<?= Job::TYPE_RESUME ?>" class="infav" title="<?= _t('', 'Удалить'); ?>"></a>
                    </div><? }
                else { ?> <div> <?
                    if(User::id()) { ?>
                        <a href="#" onclick="return app.fav('job',this);" data-id="<?= $v['id'] ?>" data-type="<?= Job::TYPE_RESUME ?>"
                           <? if(!$v['fav']){ ?>class="intofav" title="<?= _t('', 'В избранное'); ?>" <? } else { ?>class="infav" title="<?= _t('', 'Избранное'); ?>" <? } ?>></a>
                        </div><? } ?>
                <? } ?>
            </td>
        </tr>
        <? }
    }

    # ----------------------------------------------------------------------------------------------------
    # жалобы

    function getClaimReasons()
    {
        return array(
            1  => _t('', 'Неверная контактная информация'),
            2  => _t('', 'Не соответствует рубрике'),
            4  => _t('', 'Не отвечает правилам портала'),
            8  => _t('', 'Антиреклама/дискредитация'),
            32 => _t('', 'Другое'),
        );
    }

    function getClaimText($nReasons, $sComment)
    {
        $reasons = $this->getClaimReasons();
        if(!empty($nReasons) && !empty($reasons)) 
        {
            $r_text = array();
            foreach($reasons as $rk=>$rv) {
                if($rk!=32 && $rk & $nReasons) {
                    $r_text[] = $rv;
                }
            }
            $r_text = join(', ', $r_text);
            if($nReasons & 32 && !empty($sComment)) {
                $r_text .= ', '.$sComment;
            }
            return $r_text;
        }
        return '';
    }

    # ----------------------------------------------------------------------------------------------------
    # категории

    /**
     * Обрабатываем параметры категории
     */
    function categoriesProcessData($nCategoryID)
    {
        $aParams = array(
            'keyword'       => TYPE_NOTAGS,  // keyword
            'enabled'       => TYPE_BOOL,   // включена ли категория
            'main'          => TYPE_BOOL,   // выводить на главной
            'resume_mtemplate'  => TYPE_BOOL, // выводить на главной
            'vacancy_mtemplate' => TYPE_BOOL, // выводить на главной
        );
        $aData = $this->input->postm($aParams);
        $this->input->postm_lang($this->model->langCategories, $aData);

        if (Request::isPOST())
        {
            if($aData['title'][LNG] == '') {
                $this->errors->set( _t('job', 'Название категории указано некорректно') );
            }
            if($aData['keyword'] == '') {
                $this->errors->set( _t('job', 'Keyword категории указан некорректно') );
            } else if($this->isKeywordExists($aData['keyword'], TABLE_JOB_CATEGORIES, $nCategoryID) ) {
                $this->errors->set( _t('job', 'Данный keyword уже используется') );
            }
        }
        return $aData;
    }

    /**
     * Формируем список "Сфера деятельности"
     * @param integer $nSelectedID текущее значение
     * @param mixed $mEmpty не выбранное значение
     * @param boolean $bEnabled только включенные категории
     * @return string|array
     */
    function categoriesOptions($nSelectedID = 0, $mEmpty = false, $bEnabled = true)
    {
        $sql = array();
        $sql[] = '1=1';
        if($bEnabled) $sql[] = 'C.enabled = 1';
        
        $aCategories = $this->db->select('SELECT C.id, CL.title, C.keyword
                    FROM '.TABLE_JOB_CATEGORIES.' C, '.TABLE_JOB_CATEGORIES_LANG.' CL
                    WHERE '.join(' AND ', $sql).$this->db->langAnd(true, 'C', 'CL').'
                    ORDER BY C.num');
   
        $html = '';
        if(!$nSelectedID && $mEmpty===false) $mEmpty = true;
        if(!empty($mEmpty)) {
            $html .= '<option value="0">'.(is_string($mEmpty) ? $mEmpty : _t('', 'не указана')).'</option>';
        }
        foreach($aCategories as $v) {
            $html .= '<option value="'.$v['id'].'"'.($nSelectedID==$v['id']?' selected="selected"':'').' data-k="'.$v['keyword'].'">'.$v['title'].'</option>';
        }
        return $html;
    }

    /**
     * Проверка существования ключа
     * @param string $sKeyword ключ
     * @param string $sTable таблица
     * @param integer $nExceptRecordID исключая записи (ID)
     * @returns integer
     */ 
    function isKeywordExists($sKeyword, $sTable, $nExceptRecordID = null)
    {
        return $this->db->one_data('SELECT id
                               FROM '.$sTable.'
                               WHERE '.( !empty($nExceptRecordID)? ' id!='.intval($nExceptRecordID).' AND ' : '' ).'
                                  keyword = '.$this->db->str2sql($sKeyword).'  LIMIT 1');
    }

    function getDateOptions($sDate = '', $nYearOffset = 5, $nYearStart = 1990)
    {
        list($year,$month,$day) = (!empty($sDate) ? preg_split('/[\/.-]/', $sDate) : array('','',''));

        $result = array('days'   => range(1,31),
                        'months' => $this->locale->getMonthTitle(),
                        'years'  => range(date('Y') - $nYearOffset, $nYearStart) );
        $func_options = create_function('&$item,$key,$cur', '$item = "<option value=\"$item\" ".($item==$cur?"selected":"").">$item</option>";');

        array_walk($result['days'],   $func_options, $day);
        array_walk($result['months'], create_function('&$item,$key,$cur', '$item = "<option value=\"$key\" ".($key==$cur?"selected":"").">$item</option>";'), $month);
        array_walk($result['years'],  $func_options, $year);

        return array(
            'day'   => join('', $result['days']),
            'month' => join('', $result['months']),
            'year'  => join('', $result['years']));
    }

    function yandexXML()
    {
        # Вакансии
        $aItems = $this->db->select('SELECT I.id, I.link, I.created, I.modified,
                I.price, C.keyword as price_type, I.price_torg,
                JC.title as category, I.title,
                R.title_'.LNG.' as city,
                I.company_title, I.company_email, I.company_phone, I.company_contact,
                I.company_staffagency
            FROM '.TABLE_JOB_VACANCY.' I
                 INNER JOIN '.TABLE_JOB_CATEGORIES.' JC ON I.cat_id = JC.id
                 INNER JOIN '.TABLE_REGIONS.' R ON I.city_id = R.id
                 INNER JOIN '.TABLE_CURRENCIES.' C ON I.price_curr = C.id
            WHERE I.status = '.self::STATUS_PUBLICATED.' AND I.moderated > 0
            ORDER BY I.id DESC
        ');

        if( empty($aItems) ) return;

        $dateFormat = 'Y-m-d G:i:s O';
        $aVacancies = array();
        foreach($aItems as $v)
        {
            # URL страницы с объявлением.
            $sVacancy  = '<url>'.htmlspecialchars('http:'.static::url('view', $v['link'])).'</url>';
            # Дата создания объявления.
            $sVacancy .= '<creation-date>'.htmlspecialchars(date($dateFormat, strtotime($v['created']))).'</creation-date>';
            # Дата обновления объявления.
            $sVacancy .=  '<update-date>'.htmlspecialchars(date($dateFormat, strtotime($v['modified']))).'</update-date>';
            if( $v['price'] > 0 ) {
                # Зарплата: «от x1 до x2», «x1 — x2», «от x1», «до x2»
                $sVacancy .= '<salary>'.htmlspecialchars('от '.$v['price']).'</salary>';
                # Валюта, в которой измеряется зарплата (RUR, USD, EUR ...).
                $sVacancy .= '<currency>'.htmlspecialchars($v['price_type']).'</currency>';
            }
            # Область работы и специализация.
            $sVacancy .= '<category><industry>'.htmlspecialchars($v['category']).'</industry></category>';
            # Название должности.
            $sVacancy .= '<job-name>'.htmlspecialchars($v['title']).'</job-name>';
            # Адрес места работы
            $sVacancy .= '<addresses><address><location>'.htmlspecialchars(Geo::countryData('title').', '.$v['city']).'</location></address></addresses>';
            # Информация о компании, предоставляющей вакансию.
            $sVacancy .= '<company>';
                # Название компании.
                $sVacancy .= '<name>'.htmlspecialchars($v['company_title']).'</name>';
                # Адрес электронной почты компании.
                $sVacancy .= '<email>'.htmlspecialchars($v['company_email']).'</email>';
                # Телефон.
                $sVacancy .= '<phone>'.htmlspecialchars($v['company_phone']).'</phone>';
                # Контактное лицо.
                $sVacancy .= '<contact-name>'.htmlspecialchars($v['company_contact']).'</contact-name>';
                # Если компания является кадровым агентством, то указывается true, в противном случае — false
                $sVacancy .= '<hr-agency>'.htmlspecialchars(($v['company_staffagency']?'true':'false')).'</hr-agency>';
            $sVacancy .= '</company>';

            $aVacancies[] = '<vacancy>'.$sVacancy.'</vacancy>';
        }

        $sXML = strtr('<?xml version="1.0" encoding="utf-8"?>
            <source creation-time="{created}" host="{host}">
                <vacancies>{vacancies}</vacancies>
            </source>', array(
            '{created}' => date($dateFormat),
            '{host}'    => SITEHOST,
            '{vacancies}'  => join('', $aVacancies),
        ));

        Files::putFileContent(PATH_PUBLIC.'rss/yandex-vacancy.xml', $sXML);
    }

    # --------------------------------------------------------
    # Активация услуг

    function svcActivate($nItemID, $nSvcID, $aSvcData = false, array &$aSvcSettings = array())
    {
        $svc = $this->svc();
        if( ! $nSvcID ) {
            $this->errors->set(_t('svc', 'Неудалось активировать услугу'));
            return false;
        }
        if( empty($aSvcData) ) {
            $aSvcData = $svc->model->svcData($nSvcID);
            if( empty($aSvcData) ) {
                $this->errors->set(_t('svc', 'Неудалось активировать услугу'));
                return false;
            }
        }

        # получаем данные об объявлении
        if( empty($aItemData) ) {
            $aItemData = $this->model->itemData($this->typeBySvc($nSvcID), $nItemID, array(
                'id','status',// ID, статус
                'publicated_to', // дата окончания публикации
                'svc', // битовое поле активированных услуг
                'marked_to', // дата окончания "Выделение"
                'fixed_to', // дата окончания "Закрепления"
            ));
        }

        # проверяем статус объявления
        if( empty($aItemData) || $aItemData['status'] != self::STATUS_PUBLICATED ) {
            $this->errors->set( _t('job', 'Для указанного объявления невозможно активировать данную услугу') );
            return false;
        }

        # активируем услугу
        return $this->svcActivateService($nItemID, $nSvcID, $aSvcData, $aItemData, $aSvcSettings);
    }

    /**
     * Активация услуги для объявления
     * @param integer $nItemID ID объявления
     * @param integer $nSvcID ID услуги
     * @param mixed $aSvcData данные об услуге(*) или FALSE
     * @param mixed $aItemData @ref данные об объявлении или FALSE
     * @param array $aSvcSettings @ref дополнительные параметры услуги/нескольких услуг
     * @return boolean true - услуга успешно активирована, false - ошибка активации услуги
     */
    protected function svcActivateService($nItemID, $nSvcID, $aSvcData = false, &$aItemData = false, array &$aSvcSettings = array())
    {
        if( empty($nItemID) || empty($aItemData) || empty($nSvcID) ) {
            $this->errors->set(_t('svc', 'Неудалось активировать услугу'));
            return false;
        }
        $svc = $this->svc();
        if( empty($aSvcData) ) {
            $aSvcData = $svc->model->svcData($nSvcID);
            if( empty($aSvcData) ) {
                $this->errors->set(_t('svc', 'Неудалось активировать услугу'));
                return false;
            }
        }

        $sNow = $this->db->now();
        $nDay = 86400;
        $publicatedTo = strtotime($aItemData['publicated_to']);
        $aUpdate = array();
        switch ($nSvcID)
        {
            case self::SERVICE_RESUME_UP: // "поднятие"
            case self::SERVICE_VACANCY_UP: // "поднятие"
            {
                $aUpdate['publicated_order'] = $sNow;
            } break;
            case self::SERVICE_RESUME_MARK: // "выделение"
            case self::SERVICE_VACANCY_MARK: // "выделение"
            {
                $nDays = 7; // период действия услуги (в днях)

                // считаем дату окончания действия услуги
                $to = strtotime('+'.$nDays.' days', (
                        // если услуга уже активна => продлеваем срок действия
                    ($aItemData['svc'] & $nSvcID) ? strtotime($aItemData['marked_to']) :
                        // если неактивна => активируем на требуемый период от текущей даты
                        time()
                    ));
                $toStr = date('Y-m-d H:i:s', $to);
                // в случае если дата публикация объявления завершается раньше окончания услуги:
                if($publicatedTo < $to) {
                    // продлеваем публикацию
                    $aUpdate['publicated_to'] = $toStr;
                }
                // помечаем срок действия услуги
                $aUpdate['marked_to'] = $toStr;
                // помечаем активацию услуги
                $aUpdate['svc'] = ($aItemData['svc'] | $nSvcID);
            } break;
            case self::SERVICE_RESUME_FIX: // "закрепление"
            case self::SERVICE_VACANCY_FIX: // "закрепление"
            {
                $nDays = 7; // период действия услуги (в днях)

                // считаем дату окончания действия услуги
                $to = strtotime('+'.$nDays.' days', (
                        // если услуга уже активна => продлеваем срок действия
                    ($aItemData['svc'] & $nSvcID) ? strtotime($aItemData['fixed_to']) :
                        // если неактивна => активируем на требуемый период от текущей даты
                        time()
                    ));
                $toStr = date('Y-m-d H:i:s', $to);
                // в случае если дата публикация объявления завершается раньше окончания услуги:
                if($publicatedTo < $to) {
                    // продлеваем публикацию
                    $aUpdate['publicated_to'] = $toStr;
                }
                // помечаем срок действия услуги
                $aUpdate['fixed_to'] = $toStr;
                // ставим выше среди закрепленных
                $aUpdate['fixed_order'] = $sNow;
                // помечаем активацию услуги
                $aUpdate['svc'] = ($aItemData['svc'] | $nSvcID);
            } break;
        }

        $res = $this->model->itemSave($this->typeBySvc($nSvcID), $nItemID, $aUpdate);
        if( ! empty($res) ) {
            // актуализируем данные об объявлении
            // для корректной пакетной активации услуг
            if( ! empty($aUpdate) ) {
                foreach($aUpdate as $k=>$v) {
                    $aItemData[$k] = $v;
                }
            }
            return true;
        }
        return false;
    }

    function svcBillDescription($nItemID, $nSvcID, $aData = false, array &$aSvcSettings = array())
    {
        $aSvc = ( ! empty($aData['svc']) ? $aData['svc'] : $this->svc()->model->svcData($nSvcID) );
        $nTypeID = $this->typeBySvc($nSvcID);

        $aItemData = $this->model->itemData($nTypeID, $nItemID, array('link'));

        $sClass = 'bill-job-'.($nTypeID == self::TYPE_VACANCY ? 'vacancy' : 'resume').'-link';
        $sItemLink = static::url('view', $aItemData['link']);
        list($sLinkOpen, $sLinkClose) = ( ! empty($sItemLink) ? array('<a href="'.$sItemLink.'" class="'.$sClass.'" data-item="'.$nItemID.'">', '</a>') : array('','') );

        if($aSvc['type'] == Svc::TYPE_SERVICE)
        {
            switch ($nSvcID)
            {
                case self::SERVICE_RESUME_UP:    { return _t('job', 'Поднятие [a]резюме[b] в списке', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
                case self::SERVICE_RESUME_MARK:  { return _t('job', 'Выделение [a]резюме[b] цветом', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
                case self::SERVICE_RESUME_FIX:   { return _t('job', 'Закрепление [a]резюме[b]', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
                case self::SERVICE_VACANCY_UP:   { return _t('job', 'Поднятие [a]вакансии[b] в списке', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
                case self::SERVICE_VACANCY_MARK: { return _t('job', 'Выделение [a]вакансии[b] цветом', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
                case self::SERVICE_VACANCY_FIX:  { return _t('job', 'Закрепление [a]вакансии[b]', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
            }
        }
        return '';
    }

    function svcCron()
    {
        if( ! bff::cron() ) return;

        $sNow = $this->db->now();
        $sEmpty = '0000-00-00 00:00:00';

        # Деактивируем услугу "Выделение"
        $this->db->exec('UPDATE '.TABLE_JOB_RESUME.'
            SET svc = (svc - '.self::SERVICE_RESUME_MARK.'), marked_to = :empty
            WHERE (svc & '.self::SERVICE_RESUME_MARK.') AND marked_to <= :now',
            array(':now'=>$sNow,':empty'=>$sEmpty));

        $this->db->exec('UPDATE '.TABLE_JOB_VACANCY.'
            SET svc = (svc - '.self::SERVICE_VACANCY_MARK.'), marked_to = :empty
            WHERE (svc & '.self::SERVICE_VACANCY_MARK.') AND marked_to <= :now',
            array(':now'=>$sNow,':empty'=>$sEmpty));

        # Деактивируем услугу "Закрепление"
        $this->db->exec('UPDATE '.TABLE_JOB_RESUME.'
            SET svc = (svc - '.self::SERVICE_RESUME_FIX.'), fixed_to = :empty, fixed_order = :empty
            WHERE (svc & '.self::SERVICE_RESUME_FIX.') AND fixed_to <= :now',
            array(':now'=>$sNow,':empty'=>$sEmpty));

        $this->db->exec('UPDATE '.TABLE_JOB_VACANCY.'
            SET svc = (svc - '.self::SERVICE_VACANCY_FIX.'), fixed_to = :empty, fixed_order = :empty
            WHERE (svc & '.self::SERVICE_VACANCY_FIX.') AND fixed_to <= :now',
            array(':now'=>$sNow,':empty'=>$sEmpty));
    }

    function typeBySvc($nSvcID)
    {
        switch ($nSvcID){
            case self::SERVICE_RESUME_UP:
            case self::SERVICE_RESUME_MARK:
            case self::SERVICE_RESUME_FIX:
                return self::TYPE_RESUME;

            case self::SERVICE_VACANCY_UP:
            case self::SERVICE_VACANCY_MARK:
            case self::SERVICE_VACANCY_FIX:
                return self::TYPE_VACANCY;
        }
        return self::TYPE_RESUME;
    }

    /**
     * Обработка ситуации c необходимостью ре-формирования URL
     */
    public function onLinksRebuild()
    {
        $this->model->itemsLinksRebuild();
    }

    /**
     * Формирование списка директорий/файлов требующих проверки на наличие прав записи
     * @return array
     */
    public function writableCheck()
    {
        return array_merge(parent::writableCheck(), array(
            PATH_PUBLIC.'rss' => 'dir', # rss файлы
            PATH_PUBLIC.'rss'.DS.'yandex-vacancy.xml' => 'file-e', # yandex-vacancy.xml
        ));
    }
}