<?php

use bff\db\Dynprops;
use bff\utils\Files;

# таблицы
define('TABLE_REALTY_TYPES',            DB_PREFIX.'realty_types'); // типы
define('TABLE_REALTY_CATEGORIES',       DB_PREFIX.'realty_categories'); // категории
define('TABLE_REALTY_CATEGORIES_LANG',  DB_PREFIX.'realty_categories_lang'); // категории lang
define('TABLE_REALTY_CATEGORIES_TYPES', DB_PREFIX.'realty_categories_types'); // настройки типов в категориях
define('TABLE_REALTY_DYNPROPS',         DB_PREFIX.'realty_dynprops'); // дин. свойства
define('TABLE_REALTY_DYNPROPS_MULTI',   DB_PREFIX.'realty_dynprops_multi'); // дин. свойства - multi
define('TABLE_REALTY_ITEMS',            DB_PREFIX.'realty_items'); // объявления
define('TABLE_REALTY_CLAIMS',           DB_PREFIX.'realty_claims'); // жалобы
define('TABLE_REALTY_FAV',              DB_PREFIX.'realty_fav'); // избранные

abstract class RealtyBase extends Module implements IModuleWithSvc
{
    /** @var RealtyModel */
    public $model = null;
    protected $securityKey = '4e6d3def5633076776720110d633b0d1';

    # Типы дополнительных настроек цены
    const TYPE_PRICE        = 1; # цена
    const TYPE_PRICE_TORG   = 2; # торг
    const TYPE_PRICE_OBMEN  = 4; # обмен
    const TYPE_PRICE_IPOTEK = 8; # ипотека

    # Статус объявления
    const STATUS_NEW            = 1; # новое
    const STATUS_PUBLICATED     = 3; # опубликованное
    const STATUS_PUBLICATED_OUT = 4; # истекший срок публикации
    const STATUS_BLOCKED        = 5; # заблокированное

    # Тип пользователя (разместившего объявление)
    const USERTYPE_OWNER = 1; # владелец
    const USERTYPE_AGENT = 2; # агент

    # Тип сделки
    const TYPE_SALE = 1;
    const TYPE_RENT = 2;

    # ID Услуг
    const SERVICE_UP   = 32;   # поднятие
    const SERVICE_MARK = 64;   # выделенние
    const SERVICE_FIX  = 128;  # закрепление
    const SERVICE_VIP  = 4096; # vip
    
    public function init()
    {
        parent::init();

        $this->module_title = 'Недвижимость';

        bff::autoloadEx(array(
            'RealtyImages' => array('app', 'modules/realty/realty.images.php'),
        ));

        if( bff::adminPanel() ) {
            if(strpos(bff::$event, 'dynprops')!==false) {
                $this->dp();
            }
        }
    }

    /**
     * @return Realty
     */
    public static function i()
    {
        return bff::module('realty');
    }

    /**
     * @return RealtyModel
     */
    public static function model()
    {
        return bff::model('realty');
    }

    /**
     * Формирование URL
     * @param string $key ключ
     * @param mixed $opts параметры
     * @param boolean $dynamic динамическая ссылка
     * @return string
     */
    public static function url($key = '', $opts = array(), $dynamic = false)
    {
        $base = static::urlBase(LNG, $dynamic);
        switch ($key)
        {
            # Просмотр
            case 'view':
                return strtr($opts, array(
                    '{sitehost}' => SITEHOST . bff::locale()->getLanguageUrlPrefix(),
                ));
                break;
            # Поиск
            case 'search':
                return $base.'/search'.
                    ( ! empty($opts['cat']) ? '/'.$opts['cat'] : '' ).
                    ( ! empty($opts['type']) ? '/'.$opts['type'] : '').
                    ( ! empty($opts['q']) ? static::urlQuery($opts['q'], array('cat', 'type')) : '');
                break;
            # Добавление
            case 'add':
                return $base.'/add'.static::urlQuery($opts);
                break;
            # Редактирование
            case 'edit':
                return $base.'/edit'.static::urlQuery($opts);
                break;
            # Продвижение
            case 'promote':
                return $base.'/promote'.static::urlQuery($opts);
                break;
            # Агентства
            case 'agents.list':
                return $base.'/agents'.static::urlQuery($opts);
                break;
            # Главная
            case 'index':
                return $base.'/'.static::urlQuery($opts);
                break;
        }
        return $base;
    }

    public static function urlView($id, $keyword = '', $cityID = 0)
    {
        if (empty($keyword)) $keyword = 'item';
        return static::urlBase(LNG, true, array('city'=>$cityID)).'/'.$keyword.'-'.$id.'.html';
    }

    /**
     * Описание seo шаблонов страниц
     * @return array
     */
    public function seoTemplates()
    {
        $aTemplates = array(
            'pages' => array(
                'index' => array( // index
                    't'      => 'Главная страница',
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'search' => array( // search
                    't'      => 'Поиск объявлений',
                    'list'   => true,
                    'inherit'=> true,
                    'macros' => array(
                        'category' => array('t' => 'Название категории'),
                        'type' => array('t' => 'Тип'),
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'add' => array( // add
                    't'      => 'Добавление объявления',
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'view' => array( // view
                    't'      => 'Просмотр объявления',
                    'macros' => array(
                        'title'  => array('t' => 'Заголовок объявления'),
                        'description'  => array('t' => 'Адрес объявления'),
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'share_title'       => array(
                            't'    => 'Заголовок (поделиться в соц. сетях)',
                            'type' => 'text',
                        ),
                        'share_description' => array(
                            't'    => 'Описание (поделиться в соц. сетях)',
                            'type' => 'textarea',
                        ),
                        'share_sitename'    => array(
                            't'    => 'Название сайта (поделиться в соц. сетях)',
                            'type' => 'text',
                        ),
                    ),
                ),
                'agents' => array( // agents
                    't'      => 'Агентства',
                    'list'   => true,
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
            ),
        );

        return $aTemplates;
    }

    /**
     * @return RealtyImages объект
     */
    function initImages()
    {
        static $cache = null;
        if( ! isset($cache)) {
            require_once($this->module_dir.'realty.images.php');
            $cache = new RealtyImages();
        }
        return $cache;
    }    

    /**
     * Включена ли премодерация объявлений
     * @return boolean
     */
    public static function premoderation()
    {
        return (bool)config::sys('realty.premoderation', true);
    }

    /**
     * ID категории агенств
     * @return int
     */
    public static function agentsCategory()
    {
        return intval(config::sys('realty.agents.category', 112));
    }


    /**
     * Максимально доступное кол-во изображений, прикрепляемых к объявлению
     * @return integer
     */
    public static function imagesLimit()
    {
        return (int)config::sys('realty.images.limit', 8);
    }

    function preparePublicatePeriodTo($nPeriod, $nFrom)
    {
        $dateFormat = 'Y-m-d H:i:s';
        $dateWeek = (60*60 * 24 * 7);
        switch ( $nPeriod )
        {
            case 2: $dateTo = date($dateFormat, $nFrom+$dateWeek*2);  break; //2 недели
            case 3: $dateTo = date($dateFormat, $nFrom+$dateWeek*3);  break; //3 недели
            case 4: //1 месяц
           // case 5: //2 месяца
           // case 6: //3 месяца
           // case 9: //6 месяцев
            {
                $dateTo = date($dateFormat, mktime(date('H', $nFrom),date('i', $nFrom),date('s', $nFrom),date('m', $nFrom)+($nPeriod-3),date('d', $nFrom),date('Y', $nFrom)) );  
            } break; 
            case 1: default: $dateTo = date($dateFormat, $nFrom+$dateWeek);  break;   //1 неделя
        }
        return $dateTo;
    }
    
    function getPublicatePeriods()
    {
        $periods = array();
        $week = (60*60 * 24 * 7);
        $from = time();
        $periods[1] = date('d.m.Y', $from + $week);     //1 неделя
        $periods[2] = date('d.m.Y', $from + ($week*2)); //2 неделя
        $periods[3] = date('d.m.Y', $from + ($week*3)); //3 неделя
        $nM = date('m', $from); 
        $nD = date('d', $from); 
        $nY = date('Y', $from);
        $periods[4] = date('d.m.Y', mktime(0,0,0,$nM+1,$nD,$nY)); //1 месяц  
      //  $periods[5] = date('d.m.Y', mktime(0,0,0,$nM+2,$nD,$nY)); //2 месяца  
      //  $periods[6] = date('d.m.Y', mktime(0,0,0,$nM+3,$nD,$nY)); //3 месяца  
      //  $periods[9] = date('d.m.Y', mktime(0,0,0,$nM+6,$nD,$nY)); //6 месяцев
        
        $options = '<option value="1" selected="selected">'._t('', 'на 1 неделю').'</option>
                    <option value="2">'._t('', 'на 2 недели').'</option>
                    <option value="3">'._t('', 'на 3 недели').'</option>
                    <option value="4">'._t('', 'на 1 месяц').'</option>';
//                    <option value="5">на 2 месяца</option>
//                    <option value="6">на 3 месяца</option>
//                    <option value="9">на 6 месяцев</option>';
        
        return array('dates'=>$periods, 'options'=>$options);
    }

    function getItemClaimReasons()
    {
        return array(
            1  => _t('', 'Неверная контактная информация'),
            2  => _t('', 'Объявление не соответствует рубрике'),
            4  => _t('', 'Объявление не отвечает правилам портала'),
            8  => _t('', 'Некорректная фотография'),
            16 => _t('', 'Антиреклама/дискредитация'),
            32 => _t('', 'Другое'),
        );        
    }
    
    function getItemClaimText($nReasons, $sComment)
    {
        $reasons = $this->getItemClaimReasons();
        if( ! empty($nReasons) && !empty($reasons))
        {
            $r_text = array();
            foreach($reasons as $rk=>$rv) {
                if($rk!=32 && $rk & $nReasons) {
                    $r_text[] = $rv;
                }
            }
            $r_text = join(', ', $r_text);
            if($nReasons & 32 && !empty($sComment)) {
                $r_text .= ', '.$sComment;
            }
            return $r_text;
        }
        return '';
    }
    
    function itemsCounterUpdate($nCatID, $nTypeID, $bIncrement)
    {
        $act = ($bIncrement?'+ 1':'- 1');

        $this->db->exec('UPDATE '.TABLE_REALTY_CATEGORIES.'
            SET items = items '.$act.'
            WHERE id = '.$nCatID.'
        ');

        $this->db->exec('UPDATE '.TABLE_REALTY_CATEGORIES_TYPES.'
            SET items = items '.$act.'
            WHERE cat_id = '.$nCatID.' AND type_id = '.$nTypeID.'
        ');
    }
    
    function buildItemFullAddress($sAddress, $nCityID, $nDistrictID)
    {
        $sAddressFull = '<span>'.trim($sAddress, ' ,;').'</span>';
        if( $nCityID > 0 ) {
            $aRegions = $this->db->one_array('SELECT С.title_'.LNG.' as c_title, D.title_'.LNG.' as d_title
                    FROM '.TABLE_REGIONS.' С
                        LEFT JOIN '.TABLE_REGIONS_DISTRICTS.' D ON С.id = D.city_id AND D.id = :district
                    WHERE С.id = :city', array(':city'=>$nCityID, ':district'=>$nDistrictID));
            if( ! empty($aRegions)) {
                $sAddressFull .= ', '.$aRegions['c_title'].( $nDistrictID > 0 && ! empty($aRegions['d_title']) ? ', '.$aRegions['d_title'] : '' );
            }
        }
        return $sAddressFull;
    }

    function getItemTemplates($nCatID, $nTypeID)
    {
        $aData = $this->db->one_array('SELECT
            CL.tpl_descshort as descshort,
            CL.tpl_descfull  as descfull,
            CT.short_title_'.LNG.'  as title
        FROM '.TABLE_REALTY_CATEGORIES.' C,
             '.TABLE_REALTY_CATEGORIES_TYPES.' CT,
             '.TABLE_REALTY_CATEGORIES_LANG.' CL
        WHERE C.id = :cat
          AND C.id = CT.cat_id
          AND CT.type_id = :type '.$this->db->langAnd(true, 'C', 'CL'),
            array(':cat'=>$nCatID, ':type'=>$nTypeID));

        if (empty($aData)) {
            return array();
        }

        return $aData;
    }
    
    # ----------------------------------------------------------------------------------------------------
    # динамические свойства
    
    /**
     * Инициализация компонента работы с дин. свойствами
     * @return \bff\db\Dynprops объект
     */
    function dp()
    {
        static $oDp = null;
        if(isset($oDp)) return $oDp;

        $oDp = $this->attachComponent('dynprops', new Dynprops('cat_id',
                TABLE_REALTY_CATEGORIES, TABLE_REALTY_DYNPROPS, TABLE_REALTY_DYNPROPS_MULTI,
                false /* нет наследования (одноуровневые категории) */ ) );

        $oDp->setSettings(array(
            'module_name'=>'realty',
            'cache_method'=>'Realty_dpSettingsUpdated',
            'datafield_int_first'  => 1,
            'datafield_int_last'   => 14,
            'datafield_text_first' => 15,
            'datafield_text_last'  => 17,
            'typesAllowed'=>array(
                Dynprops::typeCheckboxGroup,
                Dynprops::typeRadioGroup,
                Dynprops::typeRadioYesNo,
                Dynprops::typeCheckbox,
                Dynprops::typeSelect,
                //  Dynprops::typeSelectMulti,
                Dynprops::typeInputText,
                Dynprops::typeTextarea,
                Dynprops::typeNumber,
                Dynprops::typeRange,
            ),
            'typesAllowedParent'=>array(),
            'langs' => $this->locale->getLanguages(false),
        ));
    
        return $oDp;
    }      
    
    /**
     * Получаем дин. свойства категории
     * @param integer $nCategoryID id категории
     * @param boolean $bResetCache обнулить кеш
     * @return mixed
     */
    function dpSettings($nCategoryID, $bResetCache = false)
    {                          
        if ($nCategoryID <= 0) return array();
        
        $cache = Cache::singleton($this->module_name, 'file');
        $cacheKey = 'cats-dynprops-'.LNG.'-'.$nCategoryID;
        if ($bResetCache) {
            # сбрасываем кеш настроек дин. свойств категории
            return $cache->delete($cacheKey);
        } else {
            if (($aSettings = $cache->get($cacheKey)) === false) { # ищем в кеше
                $aSettings = $this->dp()->getByOwner($nCategoryID, false, true, false);
                $cache->set($cacheKey, $aSettings); # сохраняем в кеш
            }
            return $aSettings;
        }
    }
    
    /**
     * Метод вызываемый модулем bff\db\Dynprops, в момент изменения настроек дин. свойств категории
     * @param integer $nCategoryID id категории
     * @param integer $nDynpropID id дин.свойства
     * @param string $sEvent событие, генерирующее вызов метода
     * @return mixed
     */
    function dpSettingsUpdated($nCategoryID, $nDynpropID, $sEvent)
    {
        if (empty($nCategoryID)) return false;
        $this->dpSettings($nCategoryID, true);
    }

    /**
     * Формирование формы редактирования/фильтра дин.свойств
     * @param integer $nCategoryID ID категории
     * @param boolean $bSearch формирование формы поиска
     * @param array $aData данные
     * @param array $aExtra доп. данные
     * @return string HTML template
     */
    function dpForm($nCategoryID, $bSearch = true, $aData = array(), $aExtra = array())
    {
         if (empty($nCategoryID)) return '';

         if($bSearch) {
            if( ! bff::adminPanel()) {
                $aForm = $this->dp()->form($nCategoryID, $aData, true, true, 'f', 'dynprops.search', $this->module_dir_tpl, $aExtra);
            } else {
                $aForm = $this->dp()->form($nCategoryID, $aData, true, true, 'd', 'search.inline', false, $aExtra);
            }
         } else {
            if(  ! bff::adminPanel() ) {
                $aForm = $this->dp()->form($nCategoryID, $aData, true, false, 'd', 'dynprops.form', $this->module_dir_tpl, $aExtra);
            } else {
                $aForm = $this->dp()->form($nCategoryID, $aData, true, false, 'd', 'form.table', false, $aExtra);
            }
         }
         return ( ! empty($aForm['form']) ? $aForm['form'] : '');
    }

    function dpSave($nCategoryID, array $aItemData, $sFieldname = 'd')
    {
        $aData = $this->input->post($sFieldname, TYPE_ARRAY);

        $aDynpropsData = array();
        foreach($aData as $props) {
            foreach($props as $id=>$v) {
                $aDynpropsData[$id] = $v;
            }
        }
        $dpSettings = $this->dpSettings($nCategoryID);

        // подготавливаем значения дин. свойств для сохранения
        $aResult = $this->dp()->prepareSaveDataByID($aDynpropsData, $dpSettings, 'insert/update', true);

        // полный адрес
        $sAddressFull = '';
        if( ! empty($aItemData['city_id']) ) {
            $sAddressFull = $aResult['address_full'] = $this->buildItemFullAddress($aItemData['address'], $aItemData['city_id'], $aItemData['district_id']);
        }

        // формируем описание на основе шаблонов
        $aTpl = $this->dp()->prepareTemplateByCacheKeys($aDynpropsData, $dpSettings,
                    $this->getItemTemplates($nCategoryID, $aItemData['type_id']),
                    array('{address}' => $sAddressFull) );

        return array_merge($aTpl, $aResult);
    }

    function dpPrepareSelectFieldsQuery($sPrefix = '', $nCategoryID = 0)
    {
        if (empty($nCategoryID) || $nCategoryID<0) {
            return '';
        }

        $fields = array();
        foreach($this->dpSettings($nCategoryID) as $v)
        {
            $f = $sPrefix.$this->dp()->datafield_prefix.$v['data_field'];
            if(!empty($v['cache_key'])) {
               $f .= ' as `'.$v['cache_key'].'`';
            }
             $fields[] = $f;
        }
        return (!empty($fields) ? ', ' . join(', ', $fields) : '');
    }

    # ----------------------------------------------------------------------------------------------------
    # категории
               
    /**
    * @param boolean $bEnabled только включенные категории
    * @param array $sqlFields необходимые параметры
    * @param mixed $mOptions false - возвращать массив категорий, array(sel, empty) - возвращать html <options>
    * @param boolean $bCountItems подсчитывать ли кол-во записей в категориях
    * @return mixed (string | array)
    */
    function categoriesGet($bEnabled = true, $sqlFields = array('C.*'), $mOptions = false, $bCountItems = false)
    {
        $bGetOptions = !empty($mOptions);
        $bCountItems = ($bCountItems && !$bGetOptions);

        if($bGetOptions) $sqlSelect = array('C.id','CL.title');
        else {
            $sqlSelect = ($sqlFields ? $sqlFields : array('C.*'));
            if($bCountItems) $sqlSelect[] = 'COUNT(I.id) as items';                    
        }        
        $sqlSelect = join(',', $sqlSelect);

        $sqlWhere = array();
        $sqlWhere[] = '1=1';                                                    
        if($bEnabled) $sqlWhere[] = 'C.enabled = 1';
        
        if($bCountItems) {
            $aCategories = $this->db->select('SELECT '.$sqlSelect.'
                    FROM '.TABLE_REALTY_CATEGORIES.' C 
                        LEFT JOIN '.TABLE_REALTY_ITEMS.' I ON I.cat_id = C.id
                    , '.TABLE_REALTY_CATEGORIES_LANG.' CL
                    WHERE '.join(' AND ', $sqlWhere).$this->db->langAnd(true, 'C', 'CL').'
                    GROUP BY C.id 
                    ORDER BY C.num');
        } else {
            $aCategories = $this->db->select('SELECT '.$sqlSelect.'
                    FROM '.TABLE_REALTY_CATEGORIES.' C, '.TABLE_REALTY_CATEGORIES_LANG.' CL
                    WHERE '.join(' AND ', $sqlWhere).$this->db->langAnd(true, 'C', 'CL').'
                    ORDER BY C.num');
        }


        if($bGetOptions)
        {
            $sHTML = '';   
            if( ! $mOptions['sel'] && $mOptions['empty']===false) $mOptions['empty'] = true;
            if( ! empty($mOptions['empty']))
            {
                $sHTML .= '<option value="0">'.(is_string($mOptions['empty']) ? $mOptions['empty'] : _t('', 'не указан')).'</option>';
            }
            foreach($aCategories as $v) {
                $sHTML .= '<option value="'.$v['id'].'"'.($mOptions['sel']==$v['id']?' selected="selected"':'').'>'.$v['title'].'</option>';
            }
            return $sHTML;
        } else {    
            if( ! bff::adminPanel() ) {
                return func::array_transparent($aCategories, 'id', true);
            }
            return $aCategories;
        }
    }
    
    /**
     * Проверка существования ключа
     * @param string $sKeyword ключ
     * @param string $sTable таблица
     * @param integer $nExceptRecordID исключая записи (ID)
     * @returns integer
     */ 
    function isKeywordExists($sKeyword, $sTable, $nExceptRecordID = null)
    {
        return $this->db->one_data('SELECT id
                               FROM '.$sTable.'
                               WHERE '.( ! empty($nExceptRecordID)? ' id!='.intval($nExceptRecordID).' AND ' : '' ).'
                                  keyword = '.$this->db->str2sql($sKeyword).'  LIMIT 1');
    }    
    
    function getPriceParams()
    {
        return array(
            self::TYPE_PRICE_TORG =>  _t('', 'торг'),
            self::TYPE_PRICE_OBMEN => _t('', 'обмен'),
            self::TYPE_PRICE_IPOTEK => _t('', 'возможна ипотека'),
        );
    }

    function processItemData(array &$aData)
    {
        $aData['title_alt'] = HTML::escape(strip_tags($aData['title']));
        $aData['keyword'] = mb_strtolower(func::translit($aData['title_alt']));
    }

    /**
     * Метод обрабатывающий событие "активации пользователя"
     * @param integer $nUserID ID активированного пользователя
     */
    function onUserActivated($nUserID)
    {
        /**
         * Активируем объявления, добавленные пользователем и еще неактивированные
         */
         $this->db->update(TABLE_REALTY_ITEMS, array(
            'status' => self::STATUS_PUBLICATED,
            'status_prev' => self::STATUS_NEW,
         ), array(
            'user_id' => $nUserID,
            'status'  => self::STATUS_NEW,
         ));
    }

    function deleteItem($nItemID, $nUserID = false)
    {
        if (empty($nItemID) || $nItemID<=0) return false;

        $aData = $this->model->itemData($nItemID, array('id', 'img', 'imgcnt', 'imgfav',
                        'user_id', 'cat_id', 'type_id'));
        if (empty($aData)) return false;
        
        if( ! empty($nUserID)) {
            if($aData['user_id']!=$nUserID) {
                return false;
            }
        }
        
        $res = $this->db->delete(TABLE_REALTY_ITEMS, array('id'=>$nItemID));

        if (empty($res)) return false;
        
        # удаляем изображения
        $aImg = ($aData['imgcnt'] && !empty($aData['img']) ? explode(',', $aData['img']) : array());
        $this->initImages()->deleteImagesFiles($nItemID, $aImg);
        
        # откручиваем счетчики кол-ва объявлений в категории/типе
        $this->itemsCounterUpdate($aData['cat_id'], $aData['type_id'], false);
        
        # удаляем в избранных
        $this->db->delete(TABLE_REALTY_FAV, array('item_id'=>$nItemID));
        
        if($aData['user_id'] > 0) {
            $this->security->userCounter('realty', -1, ($nUserID!==false ? false : $aData['user_id']));
        }
        
        return true;
    }
    
    # ----------------------------------------------------------------------------------------------------
    # типы

    /**
     * Обрабатываем параметры типа
     */
    function typesProcessData()
    {
        $aParams = array(              
            'keyword' => TYPE_NOTAGS,  // keyword
            'enabled' => TYPE_BOOL,    // включен ли тип
        );
        $aData = $this->input->postm($aParams);
        $this->input->postm_lang($this->model->langTypes, $aData);

        if (Request::isPOST())
        {
            if($aData['title'][LNG] == '') {
                $this->errors->set( _t('realty', 'Название типа указано некорректно') );
            }
            if($aData['keyword'] == '') {
                $this->errors->set( _t('realty', 'Keyword типа указан некорректно') );
            }            
        }
        return $aData;
    }

    /**
    * @param boolean $bEnabled только включенные типы
    * @param array $sqlFields необходимые параметры 
    * @param mixed $mOptions false - возвращать массив типов, array(sel, empty) - возвращать html <options>
    * @return mixed (string | array)
    */
    function typesGet($bEnabled = true, $sqlFields = array('T.*'), $mOptions = false)
    {
        $bGetOptions = !empty($mOptions);

        if($bGetOptions) $sqlSelect = array('T.id','T.keyword','T.title_'.LNG.' AS title');
        else { 
            $sqlSelect = ($sqlFields ? $sqlFields : array('T.*'));
        }
        $sqlSelect = join(',', $sqlSelect);

        $sqlWhere = array('1=1');
        if($bEnabled) $sqlWhere[] = 'T.enabled = 1';
        
        $aTypes = $this->db->select('SELECT '.$sqlSelect.'
                FROM '.TABLE_REALTY_TYPES.' T
                WHERE '.join(' AND ', $sqlWhere).'
                ORDER BY T.num');

        if($bGetOptions)
        {
            $sHTML = '';   
            if( ! $mOptions['sel'] && $mOptions['empty']===false) $mOptions['empty'] = true;
            if( ! empty($mOptions['empty']))
            {
                $sHTML .= '<option value="0">'.(is_string($mOptions['empty']) ? $mOptions['empty'] : _t('', 'не указан')).'</option>';
            }
            foreach($aTypes as $v) {
                $sHTML .= '<option value="'.$v['id'].'"'.($mOptions['sel']==$v['id']?' selected="selected"':'').'>'.$v['title'].'</option>';
            }
            return $sHTML;
        } else {    
            return ( ! bff::adminPanel() ? func::array_transparent($aTypes, 'id', true) : $aTypes);
        }
    }

    # --------------------------------------------------------
    # Активация услуг

    function svcActivate($nItemID, $nSvcID, $aSvcData = false, array &$aSvcSettings = array())
    {
        $svc = $this->svc();
        if( ! $nSvcID ) {
            $this->errors->set(_t('svc', 'Неудалось активировать услугу'));
            return false;
        }
        if( empty($aSvcData) ) {
            $aSvcData = $svc->model->svcData($nSvcID);
            if( empty($aSvcData) ) {
                $this->errors->set(_t('svc', 'Неудалось активировать услугу'));
                return false;
            }
        }

        // получаем данные об объявлении
        if( empty($aItemData) ) {
            $aItemData = $this->model->itemData($nItemID, array(
                            'id','status',// ID, статус
                            'publicated_to', // дата окончания публикации
                            'svc', // битовое поле активированных услуг
                            'marked_to', // дата окончания "Выделение"
                            'fixed_to', // дата окончания "Закрепления"
                            'vip_to', // дата окончания "VIP"
                            ));
        }

        // проверяем статус объявления
        if( empty($aItemData) || $aItemData['status'] != self::STATUS_PUBLICATED ) {
            $this->errors->set( _t('realty', 'Для указанного объявления невозможно активировать данную услугу') );
            return false;
        }

        // активируем услугу
        return $this->svcActivateService($nItemID, $nSvcID, $aSvcData, $aItemData, $aSvcSettings);
    }

    /**
     * Активация услуги для объявления
     * @param integer $nItemID ID объявления
     * @param integer $nSvcID ID услуги
     * @param mixed $aSvcData данные об услуге(*) или FALSE
     * @param mixed $aItemData @ref данные об объявлении или FALSE
     * @param array $aSvcSettings @ref дополнительные параметры услуги/нескольких услуг
     * @return boolean true - услуга успешно активирована, false - ошибка активации услуги
     */
    protected function svcActivateService($nItemID, $nSvcID, $aSvcData = false, &$aItemData = false, array &$aSvcSettings = array())
    {
        if( empty($nItemID) || empty($aItemData) || empty($nSvcID) ) {
            $this->errors->set(_t('svc', 'Неудалось активировать услугу'));
            return false;
        }
        $svc = $this->svc();
        if( empty($aSvcData) ) {
            $aSvcData = $svc->model->svcData($nSvcID);
            if( empty($aSvcData) ) {
                $this->errors->set(_t('svc', 'Неудалось активировать услугу'));
                return false;
            }
        }

        $sNow = $this->db->now();
        $publicatedTo = strtotime($aItemData['publicated_to']);
        $aUpdate = array();
        switch ($nSvcID)
        {
            case self::SERVICE_UP: // "поднятие"
            {
                $aUpdate['publicated_order'] = $sNow;
            } break;
            case self::SERVICE_MARK: // "выделение"
            {
                $nDays = 7; // период действия услуги (в днях)

                // считаем дату окончания действия услуги
                $to = strtotime('+'.$nDays.' days', (
                                // если услуга уже активна => продлеваем срок действия
                                ($aItemData['svc'] & $nSvcID) ? strtotime($aItemData['marked_to']) :
                                // если неактивна => активируем на требуемый период от текущей даты
                                time()
                            ));
                $toStr = date('Y-m-d H:i:s', $to);
                // в случае если дата публикация объявления завершается раньше окончания услуги:
                if($publicatedTo < $to) {
                    // продлеваем публикацию
                    $aUpdate['publicated_to'] = $toStr;
                }
                // помечаем срок действия услуги
                $aUpdate['marked_to'] = $toStr;
                // помечаем активацию услуги
                $aUpdate['svc'] = ($aItemData['svc'] | $nSvcID);
            } break;
            case self::SERVICE_FIX: // "закрепление"
            {
                $nDays = 7; // период действия услуги (в днях)

                // считаем дату окончания действия услуги
                $to = strtotime('+'.$nDays.' days', (
                                // если услуга уже активна => продлеваем срок действия
                                ($aItemData['svc'] & $nSvcID) ? strtotime($aItemData['fixed_to']) :
                                // если неактивна => активируем на требуемый период от текущей даты
                                time()
                            ));
                $toStr = date('Y-m-d H:i:s', $to);
                // в случае если дата публикация объявления завершается раньше окончания услуги:
                if($publicatedTo < $to) {
                    // продлеваем публикацию
                    $aUpdate['publicated_to'] = $toStr;
                }
                // помечаем срок действия услуги
                $aUpdate['fixed_to'] = $toStr;
                // ставим выше среди закрепленных
                $aUpdate['fixed_order'] = $sNow;
                // помечаем активацию услуги
                $aUpdate['svc'] = ($aItemData['svc'] | $nSvcID);
            } break;
            case self::SERVICE_VIP: // "vip"
            {
                $nDays = 7; // период действия услуги (в днях)

                // считаем дату окончания действия услуги
                $to = strtotime('+'.$nDays.' days', (
                                // если услуга уже активна => продлеваем срок действия
                                ($aItemData['svc'] & $nSvcID) ? strtotime($aItemData['vip_to']) :
                                // если неактивна => активируем на требуемый период от текущей даты
                                time()
                            ));
                $toStr = date('Y-m-d H:i:s', $to);
                // в случае если дата публикация объявления завершается раньше окончания услуги:
                if($publicatedTo < $to) {
                    // продлеваем публикацию
                    $aUpdate['publicated_to'] = $toStr;
                }
                // помечаем срок действия услуги
                $aUpdate['vip_to'] = $toStr;
                // помечаем активацию услуги
                $aUpdate['svc'] = ($aItemData['svc'] | $nSvcID);
            } break;
        }

        $res = $this->model->itemSave($nItemID, $aUpdate);
        if( ! empty($res) ) {
            // актуализируем данные об объявлении
            // для корректной пакетной активации услуг
            if( ! empty($aUpdate) ) {
                foreach($aUpdate as $k=>$v) {
                    $aItemData[$k] = $v;
                }
            }
            return true;
        }
        return false;
    }

    function svcBillDescription($nItemID, $nSvcID, $aData = false, array &$aSvcSettings = array())
    {
        $aSvc = ( ! empty($aData['svc']) ? $aData['svc'] :
                    $this->svc()->model->svcData($nSvcID) );

        $aItemData = $this->model->itemData($nItemID, 'link');

        $sItemLink = static::url('view', $aItemData['link']);
        list($sLinkOpen, $sLinkClose) = ( ! empty($sItemLink) ? array('<a href="'.$sItemLink.'" class="bill-realty-item-link" data-item="'.$nItemID.'">', '</a>') : array('','') );

        if($aSvc['type'] == Svc::TYPE_SERVICE)
        {
            switch ($nSvcID)
            {
                case self::SERVICE_UP:   { return _t('realty', 'Поднятие [a]объявления[b] в списке', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
                case self::SERVICE_MARK: { return _t('realty', 'Выделение [a]объявления[b] цветом', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
                case self::SERVICE_FIX:  { return _t('realty', 'Закрепление [a]объявления[b]', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
                case self::SERVICE_VIP:  { return _t('realty', 'VIP [a]объявление[b]', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
            }
        }
    }

    function svcIsVIPEnabled(&$aData = array())
    {
        static $cache;
        if( ! isset($cache) ) {
            $cache = Svc::model()->svcData(self::SERVICE_VIP);
        }
        $aData = $cache;
        return ! empty( $cache['on'] );
    }

    function svcCron()
    {
        if( ! bff::cron() ) return;

        $sNow = $this->db->now();
        $sEmpty = '0000-00-00 00:00:00';

        # Деактивируем услугу "Выделение"
        $this->db->exec('UPDATE '.TABLE_REALTY_ITEMS.'
            SET svc = (svc - '.self::SERVICE_MARK.'), marked_to = :empty
            WHERE (svc & '.self::SERVICE_MARK.') AND marked_to <= :now',
            array(':now'=>$sNow,':empty'=>$sEmpty));

        # Деактивируем услугу "Закрепление"
        $this->db->exec('UPDATE '.TABLE_REALTY_ITEMS.'
            SET svc = (svc - '.self::SERVICE_FIX.'), fixed_to = :empty, fixed_order = :empty
            WHERE (svc & '.self::SERVICE_FIX.') AND fixed_to <= :now',
            array(':now'=>$sNow,':empty'=>$sEmpty));

        # Деактивируем услугу "VIP"
        $this->db->exec('UPDATE '.TABLE_REALTY_ITEMS.'
            SET svc = (svc - '.self::SERVICE_VIP.'), vip_to = :empty
            WHERE (svc & '.self::SERVICE_VIP.') AND vip_to <= :now',
            array(':now'=>$sNow,':empty'=>$sEmpty));
    }

    function yandexXML()
    {
        $aOffers = array();
        $sCountry = Geo::countryData('title');
        $dateFormat = DATE_ATOM;

        # Объявления в категориях: Квартира(1), Комната(2), Дом(5), Участок(6)
        $aCategories = $this->db->select('SELECT id, title FROM '.TABLE_REALTY_CATEGORIES.' WHERE id IN(1,2,5,6) ORDER BY id');
        foreach($aCategories as $c)
        {
            $catID = $c['id'];
            $aItems = $this->db->select('SELECT I.id, I.link, I.created, I.modified,
                    I.cat_id, RC.title as cat_title, RT.title_'.LNG.' as type,
                    I.city_id, R.title_'.LNG.' as city, I.address,
                    I.imgcnt, I.img, I.info,
                    I.user_name, I.user_phone, I.user_email, I.user_type,
                    I.price, C.keyword as price_type, I.price_params
                    '.$this->dpPrepareSelectFieldsQuery('I.', $catID).'
                FROM '.TABLE_REALTY_ITEMS.' I
                     INNER JOIN '.TABLE_REALTY_CATEGORIES.' RC ON I.cat_id = RC.id
                     INNER JOIN '.TABLE_REALTY_TYPES.' RT ON I.type_id = RT.id
                     INNER JOIN '.TABLE_REGIONS.' R ON I.city_id = R.id
                     INNER JOIN '.TABLE_CURRENCIES.' C ON I.price_curr = C.id
                WHERE I.status = '.self::STATUS_PUBLICATED.' AND I.cat_id = :catID
                ORDER BY I.id DESC
            ', array(':catID'=>$catID));

            if( empty($aItems) ) continue;
            foreach($aItems as $v)
            {
                # Тип сделки («продажа», «аренда»)
                $sOffer  = '<type>'.htmlspecialchars(mb_strtolower($v['type'])).'</type>';
                # Тип недвижимости (рекомендуемое значение — «жилая»).
                $sOffer .= '<property-type>'.htmlspecialchars('жилая').'</property-type>';
                # Категория объекта (только: квартира, комната, дом, участок)
                $sOffer .= '<category>'.htmlspecialchars(mb_strtolower($v['cat_title'])).'</category>';
                # URL страницы с объявлением.
                $sOffer .= '<url>'.htmlspecialchars( static::url('view', $v['link']) ).'</url>';
                # Дата создания объявления.
                $sOffer .= '<creation-date>'.htmlspecialchars(date($dateFormat, strtotime($v['created']))).'</creation-date>';
                # Дата последнего обновления объявления.
                $sOffer .= '<last-update-date>'.htmlspecialchars(date($dateFormat, strtotime($v['modified']))).'</last-update-date>';
                # Местоположение объекта:
                $sOffer .= '<location>';
                    $sOffer .= '<country>'.htmlspecialchars($sCountry).'</country>';
                    # $sOffer .= '<region>'.htmlspecialchars('Рерион').'</region>';
                    $sOffer .= '<locality-name>'.htmlspecialchars($v['city']).'</locality-name>';
                    $sOffer .= '<address>'.htmlspecialchars($v['address']).'</address>';
                    $sOffer .= '</location>';
                # Информация о продавце:
                $sOffer .= '<sales-agent>';
                    $sOffer .= '<category>'.htmlspecialchars( ( $v['user_type'] == self::USERTYPE_OWNER ? 'владелец' : 'агентство') ).'</category>';
                    $sOffer .= '<name>'.htmlspecialchars($v['user_name']).'</name>';
                    $sOffer .= '<phone>'.htmlspecialchars($v['user_phone']).'</phone>';
                    $sOffer .= '<email>'.htmlspecialchars($v['user_email']).'</email>';
                    $sOffer .= '</sales-agent>';
                # Информация о сделке (Стоимость):
                $sOffer .= '<price>';
                    $sOffer .= '<value>'.htmlspecialchars($v['price']).'</value>';
                    $sOffer .= '<currency>'.htmlspecialchars($v['price_type']).'</currency>';
                    $sOffer .= '</price>';
                 if( $v['price_params'] & self::TYPE_PRICE_TORG ) {
                    $sOffer .= '<haggle>1</haggle>';
                 }
                 if( $v['price_params'] & self::TYPE_PRICE_IPOTEK ) {
                    $sOffer .= '<mortgage>1</mortgage>';
                 }

                # Квартира:
                if( $catID == 1 )
                {
                    # Площадь
                    $area = array(
                        # dyn-field-key => yandex-tag
                        'total_area' => 'area', # общая площадь
                        'living_space' => 'living-space', # жилая площадь
                        'kitchen_area' => 'kitchen-space', # площадь кухни
                    );
                    foreach($area as $ak=>$at) {
                        if( isset($v[$ak]) ) {
                            $sOffer .= '<'.$at.'><value>'.htmlspecialchars($v[$ak]).'</value><unit>кв.м</unit></'.$at.'>';
                        }
                    }
                    # Общее количество комнат в квартире
                    $sOffer .= '<rooms>'.htmlspecialchars($v['number_rooms']).'</rooms>';
                    # Для продажи и аренды комнат: количество комнат, участвующих в сделке
                    # - пишем все
                    $sOffer .= '<rooms-offered>'.htmlspecialchars($v['number_rooms']).'</rooms-offered>';
                    # Этаж
                    if( isset($v['floor']) ) {
                        $sOffer .= '<floor>'.htmlspecialchars($v['floor']).'</floor>';
                    }
                    # Общее количество этажей в доме
                    if( isset($v['floors']) ) {
                        $sOffer .= '<floors-total>'.htmlspecialchars($v['floors']).'</floors-total>';
                    }
                }
                # Комната:
                else if( $catID == 2 )
                {
                    # Общая площадь
                    if( isset($v['area_room']) ) {
                        $sOffer .= '<living-space><value>'.htmlspecialchars($v['area_room']).'</value><unit>кв.м</unit></living-space>';
                    }
                    # Этаж
                    if( isset($v['floor']) ) {
                        $sOffer .= '<floor>'.htmlspecialchars($v['floor']).'</floor>';
                    }
                    # Общее количество этажей в доме
                    if( isset($v['floors']) ) {
                        $sOffer .= '<floors-total>'.htmlspecialchars($v['floors']).'</floors-total>';
                    }
                }
                # Участок
                else if( $catID == 6 )
                {
                    # Площадь участка
                    if( isset($v['area_site']) ) {
                        $sOffer .= '<lot-area><value>'.htmlspecialchars($v['area_site']).'</value><unit>сот</unit></lot-area>';
                    }
                }

                if( $v['imgcnt'] > 0 ) {
                    $img = explode(',', $v['img']);
                    if( ! empty($img) ) {
                        foreach($img as $img_filename) {
                            $sOffer .= '<image>'.RealtyImages::url($v['id'], $img_filename, RealtyImages::szSmall).'</image>';
                        }
                    }
                }

                # Описание
                $sOffer .= '<description>'.htmlspecialchars($v['info']).'</description>';

                $aOffers[] = '<offer internal-id="'.$v['id'].'">'.$sOffer.'</offer>';
            }
        }

        $sXML = strtr('<?xml version="1.0" encoding="utf-8"?>
            <realty-feed xmlns="http://webmaster.yandex.ru/schemas/feed/realty/2010-06">
                <generation-date>{created}</generation-date>
                {offers}
            </realty-feed>', array(
            '{created}' => date($dateFormat),
            '{offers}'  => join('', $aOffers),
        ));

        Files::putFileContent(PATH_PUBLIC.'rss/yandex-realty.xml', $sXML);
    }

    /**
     * Обработка ситуации c необходимостью ре-формирования URL
     */
    public function onLinksRebuild()
    {
        $this->model->itemsLinksRebuild();
    }

    /**
     * Формирование списка директорий/файлов требующих проверки на наличие прав записи
     * @return array
     */
    public function writableCheck()
    {
        return array_merge(parent::writableCheck(), array(
            bff::path('realty', 'images') => 'dir', # изображения
            PATH_PUBLIC.'rss'             => 'dir', # rss файлы
            PATH_PUBLIC.'rss'.DS.'yandex-realty.xml' => 'file-e', # yandex-afisha.xml
        ));
    }
}